<?php
/**
 * Secure base configuration for x-ui-Bot
 *
 * Do NOT hardcode real secrets here. Configure via environment variables
 * or replace the placeholders below with private values in your deployment.
 */

error_reporting(E_ALL);
ini_set('display_errors', '0');

// --- Load .env if present ---
(function () {
    $envFile = __DIR__ . DIRECTORY_SEPARATOR . '.env';
    if (!is_readable($envFile)) return;
    $lines = file($envFile, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
    foreach ($lines as $line) {
        $line = trim($line);
        if ($line === '' || $line[0] === '#' || $line[0] === ';') continue;
        if (stripos($line, 'export ') === 0) $line = trim(substr($line, 7));
        $pos = strpos($line, '=');
        if ($pos === false) continue;
        $name = rtrim(substr($line, 0, $pos));
        $value = ltrim(substr($line, $pos + 1));
        // remove surrounding quotes if any
        if ((strlen($value) >= 2) && (($value[0] === '"' && substr($value, -1) === '"') || ($value[0] === "'" && substr($value, -1) === "'"))) {
            $value = substr($value, 1, -1);
        }
        // set env
        if ($name !== '') {
            putenv($name . '=' . $value);
            $_ENV[$name] = $value;
            $_SERVER[$name] = $value;
        }
    }
})();

// --- Security toggles ---
define('DISABLE_SSL_VERIFY', filter_var(getenv('DISABLE_SSL_VERIFY') ?: 'false', FILTER_VALIDATE_BOOLEAN));
define('BYTES_PER_GB', 1073741824);

// --- Telegram ---
$env_bot_token = getenv('BOT_TOKEN');
define('BOT_TOKEN', $env_bot_token !== false ? $env_bot_token : '');

// --- Database ---
define('DB_HOST', getenv('DB_HOST') ?: '');
define('DB_USER', getenv('DB_USER') ?: '');
define('DB_PASS', getenv('DB_PASS') ?: '');
define('DB_NAME', getenv('DB_NAME') ?: '');

// --- Payments ---
define('PAYMENT_MERCHANT_ID', getenv('PAYMENT_MERCHANT_ID') ?: '');
define('PAYMENT_CALLBACK_BASE', getenv('PAYMENT_CALLBACK_BASE') ?: ''); // Example: https://example.com

// --- Admin panel login ---
define('ADMIN_USERNAME', getenv('ADMIN_USERNAME') ?: 'admin');
define('ADMIN_PASSWORD_HASH', getenv('ADMIN_PASSWORD_HASH') ?: ''); // bcrypt hash required

// --- Telegram admin id (optional) ---
define('ADMIN_ID', (int)(getenv('ADMIN_ID') ?: 0));

// --- DB connection helper ---
function db_connect() {
    if (DB_HOST === '' || DB_USER === '' || DB_NAME === '') {
        http_response_code(500);
        exit('Database configuration missing. Set DB_* environment variables.');
    }
    $mysqli = @new mysqli(DB_HOST, DB_USER, DB_PASS, DB_NAME);
    if ($mysqli->connect_errno) {
        error_log('DB connect error: ' . $mysqli->connect_error);
        http_response_code(500);
        exit('Database connection failed.');
    }
    $mysqli->set_charset('utf8mb4');
    return $mysqli;
}

// --- Minimal runtime checks (non-fatal) ---
if (php_sapi_name() !== 'cli') {
    if (BOT_TOKEN === '') {
        error_log('BOT_TOKEN is not configured');
    }
}
