<?php
// Utility functions for managing per-server custom domains

/**
 * Ensure the servers table has a custom_domain column for overriding subscription domains.
 */
function ensure_server_domain_column(mysqli $connect): void {
    $res = $connect->query("SHOW COLUMNS FROM servers LIKE 'custom_domain'");
    if ($res && $res->num_rows === 0) {
        $connect->query("ALTER TABLE servers ADD COLUMN custom_domain VARCHAR(255) NULL");
    }
    try { $connect->query("ALTER TABLE servers ADD COLUMN custom_domain_pool_enabled TINYINT(1) NOT NULL DEFAULT 0"); } catch (Throwable $e) {}
    try { $connect->query("ALTER TABLE servers ADD COLUMN custom_domain_pool_offset INT NOT NULL DEFAULT 0"); } catch (Throwable $e) {}
}

/**
 * Ensure services table can persist per-service assigned domains.
 */
function ensure_service_domain_column(mysqli $connect): void {
    try { $connect->query("ALTER TABLE services ADD COLUMN assigned_domain VARCHAR(255) NULL"); } catch (Throwable $e) {}
}

/**
 * Parse a raw custom domain string into a cleaned array of domains.
 */
function parse_custom_domain_list(string $raw): array {
    $parts = array_map('trim', explode('|', $raw));
    $parts = array_filter($parts, function($item) {
        return $item !== '';
    });
    return array_values(array_unique($parts));
}

/**
 * Build a URL string from parse_url parts.
 */
function build_url_from_parts(array $parts): string {
    $scheme   = isset($parts['scheme']) ? $parts['scheme'] . '://' : '';
    $user     = $parts['user'] ?? '';
    $pass     = isset($parts['pass']) ? ':' . $parts['pass'] : '';
    $auth     = $user !== '' ? $user . $pass . '@' : '';
    $host     = $parts['host'] ?? '';
    $port     = isset($parts['port']) ? ':' . $parts['port'] : '';
    $path     = $parts['path'] ?? '';
    $query    = isset($parts['query']) && $parts['query'] !== '' ? '?' . $parts['query'] : '';
    $fragment = isset($parts['fragment']) && $parts['fragment'] !== '' ? '#' . $parts['fragment'] : '';

    return $scheme . $auth . $host . $port . $path . $query . $fragment;
}

/**
 * Replace the host of a URL with the provided domain.
 */
function replace_url_host(string $url, string $domain): string {
    if (empty($domain)) return $url;
    $parsed = parse_url($url);
    if (!isset($parsed['host'])) return $url;
    $parsed['host'] = $domain;
    return build_url_from_parts($parsed);
}

/**
 * Determine the next domain to allocate for a new service on the given server.
 */
function allocate_domain_for_new_service(mysqli $connect, array &$server): ?string {
    $custom = trim((string)($server['custom_domain'] ?? ''));
    if ($custom === '') return null;
    $domains = parse_custom_domain_list($custom);
    if (empty($domains)) return null;

    $pool_enabled = !empty($server['custom_domain_pool_enabled']);
    $selected = $domains[0];

    if ($pool_enabled) {
        $offset = (int)($server['custom_domain_pool_offset'] ?? 0);
        $index = $offset % count($domains);
        $selected = $domains[$index];

        $updated = false;
        if (!empty($server['id'])) {
            $stmt = $connect->prepare("UPDATE servers SET custom_domain_pool_offset = custom_domain_pool_offset + 1 WHERE id = ?");
            if ($stmt) {
                $stmt->bind_param('i', $server['id']);
                $stmt->execute();
                $stmt->close();
                $updated = true;
            }
        } elseif (!empty($server['name'])) {
            $stmt = $connect->prepare("UPDATE servers SET custom_domain_pool_offset = custom_domain_pool_offset + 1 WHERE name = ?");
            if ($stmt) {
                $stmt->bind_param('s', $server['name']);
                $stmt->execute();
                $stmt->close();
                $updated = true;
            }
        }

        if ($updated) {
            $server['custom_domain_pool_offset'] = $offset + 1;
        }
    }

    return $selected ?: null;
}

/**
 * Replace the host of a URL with the server's custom_domain if one is set.
 *
 * @param string $url     Original URL
 * @param array  $server  Server array containing optional 'custom_domain'
 * @return string Modified URL
 */
function apply_custom_domain(?string $url, array $server, ?array $service = null): string {
    if (!$url) return '';
    $domain = '';
    if ($service && !empty($service['assigned_domain'])) {
        $domain = trim((string)$service['assigned_domain']);
    }

    if ($domain === '') {
        $custom = trim((string)($server['custom_domain'] ?? ''));
        if ($custom === '') return $url;
        if (strpos($custom, '|') !== false) {
            $domains = parse_custom_domain_list($custom);
            $domain = $domains[0] ?? '';
        } else {
            $domain = $custom;
        }
    }

    if ($domain === '') return $url;
    return replace_url_host($url, $domain);
}

/**
 * Apply custom domain based on the URL's host by looking up the server entry.
 *
 * @param mysqli $connect Database connection
 * @param string $url     Original URL
 * @return string Modified URL
 */
function apply_custom_domain_by_link(mysqli $connect, ?string $url): string {
    if (!$url) return '';

    $parsed = parse_url($url);
    $host = $parsed['host'] ?? null;

    $stmt = $connect->prepare("SELECT id, svname, assigned_domain FROM services WHERE sub = ? LIMIT 1");
    if ($stmt) {
        $stmt->bind_param('s', $url);
        $stmt->execute();
        $service_res = $stmt->get_result();
        $service = $service_res ? $service_res->fetch_assoc() : null;
        $stmt->close();
    } else {
        $service = null;
    }

    if (!$service && $host) {
        $stmt = $connect->prepare("SELECT id, svname, assigned_domain FROM services WHERE assigned_domain = ? LIMIT 1");
        if ($stmt) {
            $stmt->bind_param('s', $host);
            $stmt->execute();
            $service_res = $stmt->get_result();
            $service = $service_res ? $service_res->fetch_assoc() : null;
            $stmt->close();
        }
    }

    $server = [];
    if (!empty($service['svname'])) {
        $stmt_server = $connect->prepare("SELECT id, name, custom_domain, custom_domain_pool_enabled FROM servers WHERE name = ? LIMIT 1");
        if ($stmt_server) {
            $stmt_server->bind_param('s', $service['svname']);
            $stmt_server->execute();
            $res = $stmt_server->get_result();
            $server = $res ? $res->fetch_assoc() : [];
            $stmt_server->close();
        }
    }

    if (!empty($service)) {
        return apply_custom_domain($url, $server, $service);
    }

    if (!$host) return $url;
    $stmt = $connect->prepare("SELECT custom_domain FROM servers WHERE link LIKE ? LIMIT 1");
    if ($stmt) {
        $like = "%$host%";
        $stmt->bind_param('s', $like);
        $stmt->execute();
        $res = $stmt->get_result();
        $server = $res ? $res->fetch_assoc() : null;
        $stmt->close();
        if ($server && !empty($server['custom_domain'])) {
            return replace_url_host($url, $server['custom_domain']);
        }
    }
    return $url;
}
