<?php
require_once __DIR__ . '/includes/auth.php';
json_auth_guard();
header('Content-Type: application/json; charset=utf-8');

// --- Includes and Config ---
require_once 'config.php';
require_once '../class.php';
require_once '../lib/discounts.php';
require_once '../lib/server_domains.php';
require_once '../lib/server_metadata.php';
require_once '../lib/referrals.php';
require_once '../jdf.php';
if (file_exists('../api/sanaei.php')) {
    require_once '../api/sanaei.php';
}

// --- Database Connection ---
$connect = db_connect();
ensure_server_domain_column($connect);
ensure_server_metadata_schema($connect);
ensure_referral_schema($connect);
// Ensure columns for star-based pricing exist
try { $connect->query("ALTER TABLE plan_prices ADD COLUMN stars_price INT NOT NULL DEFAULT 0"); } catch (Throwable $e) {}
try { $connect->query("ALTER TABLE orders ADD COLUMN stars_amount INT NOT NULL DEFAULT 0"); } catch (Throwable $e) {}
// Ensure default payment messages exist
try {
    $connect->query("INSERT IGNORE INTO messages (message_key, message_value) VALUES
        ('crypto_payment_message', '💎 <b>پرداخت با رمز ارز</b>\\n\\nلطفاً مبلغ معادل <b>{amount_toman} تومان</b>{amount_usdt} را به یکی از کیف پول‌های زیر واریز نمایید:\\n\\n<code>\\n{wallets}\\n</code>\\n\\n‼️ <b>مهم:</b> پس از واریز، لطفاً <b>اسکرین‌شات از رسید پرداخت</b> را به همراه <b>لینک یا هش تراکنش (TxID)</b> در <b>کپشن همان اسکرین‌شات</b> برای ما ارسال کنید.\\n\\nسفارش شما پس از تایید تراکنش توسط ادمین، فعال خواهد شد.'),
        ('stars_payment_message', 'برای پرداخت با استارز از فاکتور ارسال‌شده استفاده کنید.')");
} catch (Throwable $e) {}

// --- Helper function to get settings from DB ---
function get_setting($connect, $key) {
    $stmt = $connect->prepare("SELECT value FROM settings WHERE `key` = ? LIMIT 1");
    $stmt->bind_param("s", $key);
    $stmt->execute();
    $result = $stmt->get_result();
    if ($row = $result->fetch_assoc()) {
        return $row['value'];
    }
    $stmt->close();
    return null;
}

// --- Main API Router ---
$action = $_POST['action'] ?? $_GET['action'] ?? null;

if ($action === null) {
    echo json_encode(['success' => false, 'message' => 'Action parameter not provided.']);
    exit;
}

switch ($action) {
    case 'get_stats': get_stats($connect); exit;
    case 'get_extended_stats': get_extended_stats_api($connect); exit;
    case 'get_order_timeline': get_order_timeline_api($connect, isset($_POST['limit']) ? (int)$_POST['limit'] : 15); exit;
    case 'get_all_users': get_all_users($connect, $_POST['page'] ?? 1, $_POST['search'] ?? '', $_POST['status'] ?? '', $_POST['server'] ?? ''); exit;
    case 'get_all_services': get_all_services(
        $connect,
        $_POST['page'] ?? 1,
        $_POST['search'] ?? '',
        $_POST['server'] ?? '',
        $_POST['status'] ?? '',
        isset($_POST['include_traffic']) ? filter_var($_POST['include_traffic'], FILTER_VALIDATE_BOOLEAN) : true
    ); exit;
    case 'get_user_details': get_user_details($connect, $_POST['identifier']); exit;
    case 'update_balance': update_balance($connect, $_POST['user_id'], $_POST['amount'], $_POST['type']); exit;
    case 'delete_service': delete_service($connect, $_POST['service_id']); exit;
    case 'get_servers': get_servers($connect); exit;
    case 'get_server_tags': get_server_tags_api($connect); exit;
    case 'create_server_tag': create_server_tag($connect, $_POST['name'] ?? '', $_POST['color'] ?? ''); exit;
    case 'delete_server_tag': delete_server_tag($connect, $_POST['tag_id'] ?? null); exit;
    case 'add_server': add_server($connect, $_POST); exit;
    case 'update_server': update_server($connect, $_POST); exit;
    case 'toggle_server_status': toggle_server_status($connect, $_POST['server_id']); exit;
    case 'delete_server': delete_server($connect, $_POST['server_id']); exit;
    case 'toggle_server_hidden': toggle_server_hidden($connect, $_POST['server_id']); exit;
    case 'set_test_server': set_test_server($connect, $_POST['server_id']); exit;
    case 'get_plans': get_plans($connect); exit;
    case 'add_plan': add_plan($connect, $_POST); exit;
    case 'delete_plan': delete_plan($connect, $_POST['plan_id']); exit;
    case 'get_prices': get_prices($connect, $_POST['server_id']); exit;
    case 'set_price': set_price($connect, $_POST); exit;
    case 'export_users_csv': export_users_csv($connect); exit;
    case 'get_messages': get_messages($connect, $_POST['search'] ?? ''); exit;
    case 'update_message': update_message($connect, $_POST['message_key'], $_POST['message_value']); exit;
    case 'broadcast': broadcast_message($connect, $_POST['message']); exit;
    case 'get_settings': get_settings($connect); exit;
    case 'update_settings': update_settings($connect, $_POST); exit;
    case 'upload_service_template': upload_service_template($connect); exit;
    case 'upload_service_template_b64': upload_service_template_b64($connect); exit;
    case 'delete_service_template': delete_service_template($connect); exit;
    // Payment thumbnail template endpoints
    case 'upload_payment_template': upload_payment_template($connect); exit;
    case 'upload_payment_template_b64': upload_payment_template_b64($connect); exit;
    case 'delete_payment_template': delete_payment_template($connect); exit;
    // Discounts
    case 'get_discounts': get_discounts($connect); exit;
    case 'create_discount': create_discount($connect, $_POST); exit;
    case 'update_discount': update_discount($connect, $_POST); exit;
    case 'delete_discount': delete_discount($connect, $_POST['id'] ?? null); exit;
    case 'toggle_discount_status': toggle_discount_status($connect, $_POST['id'] ?? null); exit;
    default:
        echo json_encode(['success' => false, 'message' => 'Invalid action.']);
    exit;
}

mysqli_close($connect);

// --- Function Definitions ---

function toggle_server_hidden($connect, $server_id) {
    $server_id = (int)$server_id;
    $stmt = $connect->prepare("UPDATE servers SET is_hidden = 1 - is_hidden WHERE id = ?");
    $stmt->bind_param("i", $server_id);
    if ($stmt->execute()) { echo json_encode(['success' => true]); } else { echo json_encode(['success' => false, 'message' => 'خطا در تغییر وضعیت نمایش.']); }
    $stmt->close();
}

function set_test_server($connect, $server_id) {
    $server_id = (int)$server_id;
    $connect->begin_transaction();
    try {
        $connect->query("UPDATE servers SET is_test_server = 0");
        $stmt = $connect->prepare("UPDATE servers SET is_test_server = 1 WHERE id = ?");
        $stmt->bind_param("i", $server_id);
        $stmt->execute();
        $stmt->close();
        $connect->commit();
        echo json_encode(['success' => true]);
    } catch (mysqli_sql_exception $exception) {
        $connect->rollback();
        echo json_encode(['success' => false, 'message' => 'خطا در انتخاب سرور تست.']);
    }
}

function broadcast_message($connect, $message) {
    if (empty($message)) { echo json_encode(['success' => false, 'message' => 'پیام نمی‌تواند خالی باشد.']); return; }
    
    $admin_id = get_setting($connect, 'admin_id');
    if (empty($admin_id)) {
        echo json_encode(['success' => false, 'message' => 'شناسه ادمین در تنظیمات یافت نشد. لطفاً ابتدا آن را تنظیم کنید.']);
        return;
    }

    $encoded_text = urlencode($message);
    $connect->query("DELETE FROM `send`");
    
    $stmt = $connect->prepare("INSERT INTO `send`(`step`, `text`, `from`, `user`) VALUES ('send', ?, ?, '0')");
    $stmt->bind_param("ss", $encoded_text, $admin_id);
    if ($stmt->execute()) {
        echo json_encode(['success' => true, 'message' => 'پیام در صف ارسال قرار گرفت.']);
    } else {
        echo json_encode(['success' => false, 'message' => 'خطا در قرار دادن پیام در صف.']);
    }
    $stmt->close();
}

function get_settings($connect) {
    $settings = [];
    $keys = ['card_info', 'join_status', 'join_channels', 'bot_status', 'admin_id', 'card_payment_status', 'crypto_payment_status', 'stars_payment_status', 'crypto_wallets', 'kyc_status', 'kyc_channel_id', 'kyc_message', 'service_template_path', 'service_image_enabled', 'payment_template_path', 'payment_image_enabled', 'referral_status', 'referral_reward_type', 'referral_reward_value', 'referral_reward_min_order', 'referral_reward_expires_days', 'referral_reward_prefix', 'referral_reward_limit'];
    $placeholders = implode(',', array_fill(0, count($keys), '?'));
    
    $stmt = $connect->prepare("SELECT `key`, `value` FROM settings WHERE `key` IN ($placeholders)");
    $stmt->bind_param(str_repeat('s', count($keys)), ...$keys);
    $stmt->execute();
    $result = $stmt->get_result();
    
    while($row = $result->fetch_assoc()){
        $settings[$row['key']] = $row['value'];
    }
    $stmt->close();

    $service_template_path = $settings['service_template_path'] ?? '';
    $service_template_url = $service_template_path ? ('../' . ltrim($service_template_path, '/')) : '';
    $payment_template_path = $settings['payment_template_path'] ?? '';
    $payment_template_url = $payment_template_path ? ('../' . ltrim($payment_template_path, '/')) : '';

    echo json_encode(['success' => true, 'settings' => [
        'card_number' => $settings['card_info'] ?? '',
        'join_status' => $settings['join_status'] ?? 'off',
        'join_channels' => $settings['join_channels'] ?? '',
        'bot_status' => $settings['bot_status'] ?? 'on',
        'admin_id' => $settings['admin_id'] ?? '',
        'card_payment_status' => $settings['card_payment_status'] ?? 'on',
        'crypto_payment_status' => $settings['crypto_payment_status'] ?? 'off',
        'stars_payment_status' => $settings['stars_payment_status'] ?? 'off',
        'crypto_wallets' => $settings['crypto_wallets'] ?? '',
        'kyc_status' => $settings['kyc_status'] ?? 'off',
        'kyc_channel_id' => $settings['kyc_channel_id'] ?? '',
        'kyc_message' => $settings['kyc_message'] ?? '',
        'service_template_url' => $service_template_url,
        'service_template_path' => $service_template_path,
        'service_image_enabled' => $settings['service_image_enabled'] ?? 'on',
        'payment_template_url' => $payment_template_url,
        'payment_template_path' => $payment_template_path,
        'payment_image_enabled' => $settings['payment_image_enabled'] ?? 'on',
        'referral_status' => $settings['referral_status'] ?? 'off',
        'referral_reward_type' => $settings['referral_reward_type'] ?? 'fixed',
        'referral_reward_value' => $settings['referral_reward_value'] ?? '0',
        'referral_reward_min_order' => $settings['referral_reward_min_order'] ?? '0',
        'referral_reward_expires_days' => $settings['referral_reward_expires_days'] ?? '0',
        'referral_reward_prefix' => $settings['referral_reward_prefix'] ?? 'REF',
        'referral_reward_limit' => $settings['referral_reward_limit'] ?? '1',
    ]]);
}

function update_settings($connect, $data) {
    try {
        $settings_to_update = [
            'card_info' => $data['card_number'] ?? '',
            'join_status' => in_array($data['join_status'] ?? '', ['on', 'off']) ? $data['join_status'] : 'off',
            'join_channels' => $data['join_channels'] ?? '',
            'bot_status' => in_array($data['bot_status'] ?? '', ['on', 'off']) ? $data['bot_status'] : 'on',
            'admin_id' => $data['admin_id'] ?? '',
            'card_payment_status' => in_array($data['card_payment_status'] ?? '', ['on', 'off']) ? $data['card_payment_status'] : 'off',
            'crypto_payment_status' => in_array($data['crypto_payment_status'] ?? '', ['on', 'off']) ? $data['crypto_payment_status'] : 'off',
            'stars_payment_status' => in_array($data['stars_payment_status'] ?? '', ['on', 'off']) ? $data['stars_payment_status'] : 'off',
            'crypto_wallets' => $data['crypto_wallets'] ?? '',
            'kyc_status' => in_array($data['kyc_status'] ?? '', ['on', 'off']) ? $data['kyc_status'] : 'off',
            'kyc_channel_id' => $data['kyc_channel_id'] ?? '',
            'kyc_message' => $data['kyc_message'] ?? '',
            'service_image_enabled' => in_array(($data['service_image_enabled'] ?? ''), ['on','off']) ? $data['service_image_enabled'] : 'on',
            'payment_image_enabled' => in_array(($data['payment_image_enabled'] ?? ''), ['on','off']) ? $data['payment_image_enabled'] : 'on',
            'referral_status' => in_array($data['referral_status'] ?? '', ['on','off']) ? $data['referral_status'] : 'off',
            'referral_reward_type' => in_array($data['referral_reward_type'] ?? '', ['fixed','percent']) ? $data['referral_reward_type'] : 'fixed',
            'referral_reward_value' => (string)max(0, (float)($data['referral_reward_value'] ?? 0)),
            'referral_reward_min_order' => (string)max(0, (int)($data['referral_reward_min_order'] ?? 0)),
            'referral_reward_expires_days' => (string)max(0, (int)($data['referral_reward_expires_days'] ?? 0)),
            'referral_reward_prefix' => ($prefix = strtoupper(substr(preg_replace('/[^A-Z0-9]/i', '', (string)($data['referral_reward_prefix'] ?? 'REF')), 0, 12))) ? $prefix : 'REF',
            'referral_reward_limit' => (string)max(1, (int)($data['referral_reward_limit'] ?? 1)),
        ];

        $stmt = $connect->prepare("INSERT INTO settings (`key`, `value`) VALUES (?, ?) ON DUPLICATE KEY UPDATE `value` = ?");
        
        foreach($settings_to_update as $key => $value) {
            $stmt->bind_param("sss", $key, $value, $value);
            $stmt->execute();
        }
        $stmt->close();
        
        echo json_encode(['success' => true]);
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => 'خطا در ذخیره تنظیمات: ' . $e->getMessage()]);
    }
}

// --- Image Upload: Service Card Template ---
function upload_service_template($connect) {
    try {
        if (!isset($_FILES['file'])) {
            // When post_max_size is exceeded, $_POST may still include small fields like action,
            // but $_FILES will be empty. Expose helpful hint.
            $maxPost = ini_get('post_max_size');
            $maxUpload = ini_get('upload_max_filesize');
            $cl = $_SERVER['CONTENT_LENGTH'] ?? '';
            $ct = $_SERVER['CONTENT_TYPE'] ?? '';
            echo json_encode([
                'success' => false,
                'message' => 'هیچ فایلی ارسال نشد. لطفاً مجدداً تلاش کنید.',
                'hint' => 'post_max_size=' . $maxPost . ', upload_max_filesize=' . $maxUpload . ', content_length=' . $cl . ', content_type=' . $ct
            ]);
            return;
        }
        if ($_FILES['file']['error'] !== UPLOAD_ERR_OK) {
            $err = (int)$_FILES['file']['error'];
            $msg = 'خطای ناشناخته در آپلود.';
            $hint = 'upload_max_filesize=' . ini_get('upload_max_filesize') . ', post_max_size=' . ini_get('post_max_size') . ', content_length=' . ($_SERVER['CONTENT_LENGTH'] ?? '');
            switch ($err) {
                case UPLOAD_ERR_INI_SIZE: $msg = 'حجم فایل بیشتر از upload_max_filesize در سرور است.'; break;
                case UPLOAD_ERR_FORM_SIZE: $msg = 'حجم فایل از محدودیت فرم فراتر رفته است.'; break;
                case UPLOAD_ERR_PARTIAL: $msg = 'فایل به طور ناقص آپلود شد.'; break;
                case UPLOAD_ERR_NO_FILE: $msg = 'فایلی انتخاب نشده است.'; break;
                case UPLOAD_ERR_NO_TMP_DIR: $msg = 'پوشه موقت سرور در دسترس نیست.'; break;
                case UPLOAD_ERR_CANT_WRITE: $msg = 'نوشتن فایل روی دیسک ناموفق بود.'; break;
                case UPLOAD_ERR_EXTENSION: $msg = 'آپلود توسط افزونه PHP متوقف شد.'; break;
            }
            echo json_encode(['success' => false, 'message' => $msg, 'hint' => $hint]);
            return;
        }

        $file = $_FILES['file'];
        $allowedMime = [
            'image/jpeg' => 'jpg',
            'image/jpg'  => 'jpg',
            'image/pjpeg'=> 'jpg',
            'image/png'  => 'png',
            'image/x-png'=> 'png',
            'image/webp' => 'webp'
        ];
        $mime = null;
        if (function_exists('finfo_open')) {
            $finfo = finfo_open(FILEINFO_MIME_TYPE);
            if ($finfo) { $mime = finfo_file($finfo, $file['tmp_name']); finfo_close($finfo); }
        }
        if ($mime === null) {
            // fallback to extension-based guess if fileinfo not available
            $extGuess = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));
            $mime = $extGuess === 'png' ? 'image/png' : ($extGuess === 'webp' ? 'image/webp' : 'image/jpeg');
        }
        if (!isset($allowedMime[$mime])) {
            echo json_encode(['success' => false, 'message' => 'فرمت تصویر مجاز نیست. JPG/PNG/WEBP مجاز است.']);
            return;
        }
        // Max ~5MB
        if ($file['size'] > 5 * 1024 * 1024) {
            echo json_encode(['success' => false, 'message' => 'حجم فایل بیش از 5 مگابایت است.']);
            return;
        }

        $ext = $allowedMime[$mime];
        $uploadsDir = realpath(__DIR__ . '/../assets') . DIRECTORY_SEPARATOR . 'uploads';
        if ($uploadsDir === false) {
            $uploadsDir = __DIR__ . '/../assets/uploads';
        }
        if (!is_dir($uploadsDir)) {
            @mkdir($uploadsDir, 0777, true);
        }
        if (!is_dir($uploadsDir) || !is_writable($uploadsDir)) {
            echo json_encode(['success' => false, 'message' => 'دسترسی نوشتن به مسیر آپلود وجود ندارد.']);
            return;
        }

        // Remove previous if exists (only if inside uploads)
        $prev = get_setting($connect, 'service_template_path');
        if ($prev) {
            $prevPath = realpath(__DIR__ . '/../' . ltrim($prev, '/\\'));
            $uploadsRoot = realpath($uploadsDir);
            if ($prevPath && $uploadsRoot && strpos($prevPath, $uploadsRoot) === 0 && is_file($prevPath)) {
                @unlink($prevPath);
            }
        }

        $basename = 'service_template_' . date('Ymd_His') . '_' . bin2hex(random_bytes(3)) . '.' . $ext;
        $targetPath = rtrim($uploadsDir, '/\\') . DIRECTORY_SEPARATOR . $basename;
        if (!is_uploaded_file($file['tmp_name'])) {
            echo json_encode(['success' => false, 'message' => 'فایل آپلود معتبر تشخیص داده نشد.']);
            return;
        }
        if (!move_uploaded_file($file['tmp_name'], $targetPath)) {
            echo json_encode(['success' => false, 'message' => 'انتقال فایل آپلود شده ناموفق بود.']);
            return;
        }

        // Save relative path from project root
        $relative = 'assets/uploads/' . $basename;
        $stmt = $connect->prepare("INSERT INTO settings (`key`, `value`) VALUES ('service_template_path', ?) ON DUPLICATE KEY UPDATE `value` = VALUES(`value`)");
        $stmt->bind_param('s', $relative);
        $stmt->execute();
        $stmt->close();

        echo json_encode(['success' => true, 'path' => $relative, 'url' => '../' . $relative]);
    } catch (Throwable $e) {
        echo json_encode(['success' => false, 'message' => 'خطای داخلی: ' . $e->getMessage()]);
    }
}

function delete_service_template($connect) {
    try {
        $prev = get_setting($connect, 'service_template_path');
        if ($prev) {
            $prevPath = realpath(__DIR__ . '/../' . ltrim($prev, '/\\'));
            $uploadsDir = realpath(__DIR__ . '/../assets/uploads');
            if ($prevPath && $uploadsDir && strpos($prevPath, $uploadsDir) === 0 && is_file($prevPath)) {
                @unlink($prevPath);
            }
        }
        // Remove setting
        $stmt = $connect->prepare("DELETE FROM settings WHERE `key` = 'service_template_path'");
        $stmt->execute();
        $stmt->close();
        echo json_encode(['success' => true]);
    } catch (Throwable $e) {
        echo json_encode(['success' => false, 'message' => 'خطای داخلی: ' . $e->getMessage()]);
    }
}

// Base64 upload fallback: avoids PHP's file-upload size limit (upload_max_filesize)
function upload_service_template_b64($connect) {
    try {
        // Accept data via application/x-www-form-urlencoded 'data' or raw JSON {data: ...}
        $payload = $_POST['data'] ?? null;
        if ($payload === null) {
            $raw = file_get_contents('php://input');
            $json = json_decode($raw, true);
            if (is_array($json) && isset($json['data'])) $payload = $json['data'];
        }
        if (!$payload || !is_string($payload)) {
            echo json_encode(['success' => false, 'message' => 'داده تصویری ارسال نشده است.']);
            return;
        }
        if (!preg_match('/^data:(image\/(?:png|jpeg|jpg|webp));base64,(.*)$/i', $payload, $m)) {
            echo json_encode(['success' => false, 'message' => 'فرمت داده معتبر نیست.']);
            return;
        }
        $mime = strtolower($m[1]);
        $b64 = $m[2];
        $allowed = [
            'image/jpeg' => 'jpg',
            'image/jpg'  => 'jpg',
            'image/png'  => 'png',
            'image/webp' => 'webp',
        ];
        if (!isset($allowed[$mime])) {
            echo json_encode(['success' => false, 'message' => 'نوع تصویر مجاز نیست.']);
            return;
        }
        $binary = base64_decode($b64, true);
        if ($binary === false) { echo json_encode(['success' => false, 'message' => 'دیکد کردن تصویر ناموفق بود.']); return; }
        // Limit ~6MB decoded to stay under memory limits
        if (strlen($binary) > 6 * 1024 * 1024) { echo json_encode(['success' => false, 'message' => 'حجم تصویر پس از فشرده‌سازی زیاد است.']); return; }

        $uploadsDir = realpath(__DIR__ . '/../assets') . DIRECTORY_SEPARATOR . 'uploads';
        if ($uploadsDir === false) { $uploadsDir = __DIR__ . '/../assets/uploads'; }
        if (!is_dir($uploadsDir)) { @mkdir($uploadsDir, 0777, true); }
        if (!is_dir($uploadsDir) || !is_writable($uploadsDir)) { echo json_encode(['success' => false, 'message' => 'دسترسی نوشتن به مسیر آپلود وجود ندارد.']); return; }

        // Remove previous if exists (only if inside uploads)
        $prev = get_setting($connect, 'service_template_path');
        if ($prev) {
            $prevPath = realpath(__DIR__ . '/../' . ltrim($prev, '/\\'));
            $uploadsRoot = realpath($uploadsDir);
            if ($prevPath && $uploadsRoot && strpos($prevPath, $uploadsRoot) === 0 && is_file($prevPath)) { @unlink($prevPath); }
        }

        $ext = $allowed[$mime];
        $basename = 'service_template_' . date('Ymd_His') . '_' . bin2hex(random_bytes(3)) . '.' . $ext;
        $targetPath = rtrim($uploadsDir, '/\\') . DIRECTORY_SEPARATOR . $basename;
        if (file_put_contents($targetPath, $binary) === false) { echo json_encode(['success' => false, 'message' => 'نوشتن فایل ناموفق بود.']); return; }

        // Save setting
        $relative = 'assets/uploads/' . $basename;
        $stmt = $connect->prepare("INSERT INTO settings (`key`, `value`) VALUES ('service_template_path', ?) ON DUPLICATE KEY UPDATE `value` = VALUES(`value`)");
        $stmt->bind_param('s', $relative);
        $stmt->execute();
        $stmt->close();

        echo json_encode(['success' => true, 'path' => $relative, 'url' => '../' . $relative, 'method' => 'base64']);
    } catch (Throwable $e) {
        echo json_encode(['success' => false, 'message' => 'خطای داخلی: ' . $e->getMessage()]);
    }
}

// --- Image Upload: Payment Thumbnail Template ---
function upload_payment_template($connect) {
    try {
        if (!isset($_FILES['file'])) {
            $maxPost = ini_get('post_max_size');
            $maxUpload = ini_get('upload_max_filesize');
            $cl = $_SERVER['CONTENT_LENGTH'] ?? '';
            $ct = $_SERVER['CONTENT_TYPE'] ?? '';
            echo json_encode(['success' => false, 'message' => 'فایل ارسال نشده است.', 'hint' => 'post_max_size=' . $maxPost . ', upload_max_filesize=' . $maxUpload . ', content_length=' . $cl . ', content_type=' . $ct]);
            return;
        }
        if ($_FILES['file']['error'] !== UPLOAD_ERR_OK) {
            echo json_encode(['success' => false, 'message' => 'خطا در آپلود فایل.']);
            return;
        }
        $file = $_FILES['file'];
        $allowed = ['image/jpeg'=>'jpg','image/jpg'=>'jpg','image/png'=>'png','image/webp'=>'webp'];
        $mime = null; if (function_exists('finfo_open')) { $f=finfo_open(FILEINFO_MIME_TYPE); if($f){$mime=finfo_file($f,$file['tmp_name']); finfo_close($f);} }
        if ($mime===null) { $extGuess = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION)); $mime = $extGuess==='png'?'image/png':($extGuess==='webp'?'image/webp':'image/jpeg'); }
        if (!isset($allowed[$mime])) { echo json_encode(['success'=>false,'message'=>'فرمت مجاز نیست (jpg/png/webp).']); return; }
        $ext = $allowed[$mime];
        $uploadsDir = realpath(__DIR__ . '/../assets') . DIRECTORY_SEPARATOR . 'uploads'; if ($uploadsDir === false) { $uploadsDir = __DIR__ . '/../assets/uploads'; }
        if (!is_dir($uploadsDir)) { @mkdir($uploadsDir, 0777, true); }
        if (!is_dir($uploadsDir) || !is_writable($uploadsDir)) { echo json_encode(['success'=>false,'message'=>'دسترسی نوشتن مسیر آپلود وجود ندارد.']); return; }
        // remove previous
        $prev = get_setting($connect, 'payment_template_path');
        if ($prev) { $prevPath = realpath(__DIR__ . '/../' . ltrim($prev,'/\\')); $uploadsRoot = realpath($uploadsDir); if ($prevPath && $uploadsRoot && strpos($prevPath, $uploadsRoot)===0 && is_file($prevPath)) { @unlink($prevPath); } }
        $basename = 'payment_template_' . date('Ymd_His') . '_' . bin2hex(random_bytes(3)) . '.' . $ext;
        $target = rtrim($uploadsDir,'/\\') . DIRECTORY_SEPARATOR . $basename;
        if (!is_uploaded_file($file['tmp_name'])) { echo json_encode(['success'=>false,'message'=>'آپلود نامعتبر.']); return; }
        if (!move_uploaded_file($file['tmp_name'], $target)) { echo json_encode(['success'=>false,'message'=>'انتقال فایل ناموفق.']); return; }
        $relative = 'assets/uploads/' . $basename;
        $stmt = $connect->prepare("INSERT INTO settings (`key`, `value`) VALUES ('payment_template_path', ?) ON DUPLICATE KEY UPDATE `value` = VALUES(`value`)");
        $stmt->bind_param('s',$relative); $stmt->execute(); $stmt->close();
        echo json_encode(['success'=>true,'url'=>'../'.$relative,'path'=>$relative]);
    } catch (Throwable $e) { echo json_encode(['success'=>false,'message'=>'خطای سرور: '.$e->getMessage()]); }
}

function delete_payment_template($connect) {
    try {
        $prev = get_setting($connect, 'payment_template_path');
        if ($prev) { $prevPath = realpath(__DIR__ . '/../' . ltrim($prev,'/\\')); $uploadsDir = realpath(__DIR__ . '/../assets/uploads'); if ($prevPath && $uploadsDir && strpos($prevPath, $uploadsDir)===0 && is_file($prevPath)) { @unlink($prevPath); } }
        $stmt = $connect->prepare("DELETE FROM settings WHERE `key` = 'payment_template_path'");
        $stmt->execute(); $stmt->close();
        echo json_encode(['success'=>true]);
    } catch (Throwable $e) { echo json_encode(['success'=>false,'message'=>'خطای سرور: '.$e->getMessage()]); }
}

function upload_payment_template_b64($connect) {
    try {
        $dataUrl = $_POST['data'] ?? '';
        if (!preg_match('#^data:image/(png|jpeg|webp);base64,#',$dataUrl,$m)) { echo json_encode(['success'=>false,'message'=>'داده تصویر نامعتبر است.']); return; }
        $ext = $m[1] === 'jpeg' ? 'jpg' : $m[1];
        $raw = base64_decode(substr($dataUrl, strpos($dataUrl, ',')+1));
        if ($raw === false) { echo json_encode(['success'=>false,'message'=>'Base64 نامعتبر.']); return; }
        $uploadsDir = realpath(__DIR__ . '/../assets') . DIRECTORY_SEPARATOR . 'uploads'; if ($uploadsDir === false) { $uploadsDir = __DIR__ . '/../assets/uploads'; }
        if (!is_dir($uploadsDir)) { @mkdir($uploadsDir, 0777, true); }
        if (!is_dir($uploadsDir) || !is_writable($uploadsDir)) { echo json_encode(['success'=>false,'message'=>'دسترسی نوشتن مسیر آپلود وجود ندارد.']); return; }
        // remove previous
        $prev = get_setting($connect, 'payment_template_path');
        if ($prev) { $prevPath = realpath(__DIR__ . '/../' . ltrim($prev,'/\\')); $uploadsRoot = realpath($uploadsDir); if ($prevPath && $uploadsRoot && strpos($prevPath, $uploadsRoot)===0 && is_file($prevPath)) { @unlink($prevPath); } }
        $basename = 'payment_template_' . date('Ymd_His') . '_' . bin2hex(random_bytes(3)) . '.' . $ext;
        $target = rtrim($uploadsDir,'/\\') . DIRECTORY_SEPARATOR . $basename;
        if (file_put_contents($target, $raw) === false) { echo json_encode(['success'=>false,'message'=>'ذخیره فایل ناموفق بود.']); return; }
        $relative = 'assets/uploads/' . $basename;
        $stmt = $connect->prepare("INSERT INTO settings (`key`, `value`) VALUES ('payment_template_path', ?) ON DUPLICATE KEY UPDATE `value` = VALUES(`value`)");
        $stmt->bind_param('s',$relative); $stmt->execute(); $stmt->close();
        echo json_encode(['success'=>true,'url'=>'../'.$relative,'path'=>$relative,'method'=>'base64']);
    } catch (Throwable $e) { echo json_encode(['success'=>false,'message'=>'خطای سرور: '.$e->getMessage()]); }
}

// -------------------- DISCOUNTS --------------------
function normalize_discount_payload(array $data): array {
    $code = strtoupper(trim($data['code'] ?? ''));
    $type = ($data['type'] ?? 'percent') === 'fixed' ? 'fixed' : 'percent';
    $value = (float)($data['value'] ?? 0);
    $max_uses = isset($data['max_uses']) && $data['max_uses'] !== '' ? (string)(int)$data['max_uses'] : '';
    $per_user_limit = isset($data['per_user_limit']) && $data['per_user_limit'] !== '' ? (string)(int)$data['per_user_limit'] : '';
    $min_order_amount = isset($data['min_order_amount']) && $data['min_order_amount'] !== '' ? (string)(int)$data['min_order_amount'] : '';
    $expires_at = trim($data['expires_at'] ?? '');
    $expires_at = $expires_at !== '' ? $expires_at : '';
    $is_active = isset($data['is_active']) ? (int)(($data['is_active'] === 'on' || $data['is_active'] == 1) ? 1 : 0) : 1;
    $note = trim($data['note'] ?? '');
    return compact('code','type','value','max_uses','per_user_limit','min_order_amount','expires_at','is_active','note');
}

function get_discounts($connect) {
    ensure_discount_schema($connect);
    $res = $connect->query("SELECT id, code, type, value, max_uses, used_count, per_user_limit, min_order_amount, expires_at, is_active, note, created_at FROM discount_codes ORDER BY id DESC");
    $list = [];
    while ($row = $res->fetch_assoc()) $list[] = $row;
    echo json_encode(['success' => true, 'discounts' => $list]);
}

function create_discount($connect, $data) {
    ensure_discount_schema($connect);
    $p = normalize_discount_payload($data);
    if ($p['code'] === '') { echo json_encode(['success' => false, 'message' => 'کد اجباری است.']); return; }
    $stmt = $connect->prepare("INSERT INTO discount_codes (code, type, value, max_uses, per_user_limit, min_order_amount, expires_at, is_active, note)
                               VALUES (?, ?, ?, NULLIF(?, ''), NULLIF(?, ''), NULLIF(?, ''), NULLIF(?, ''), ?, ?)");
    $stmt->bind_param('ssdssssis', $p['code'], $p['type'], $p['value'], $p['max_uses'], $p['per_user_limit'], $p['min_order_amount'], $p['expires_at'], $p['is_active'], $p['note']);
    $ok = $stmt->execute();
    $err = $stmt->error;
    $stmt->close();
    echo json_encode(['success' => $ok, 'message' => $ok ? 'ایجاد شد' : ('خطا: ' . $err)]);
}

function update_discount($connect, $data) {
    ensure_discount_schema($connect);
    $id = (int)($data['id'] ?? 0);
    if ($id <= 0) { echo json_encode(['success' => false, 'message' => 'شناسه نامعتبر.']); return; }
    $p = normalize_discount_payload($data);
    if ($p['code'] === '') { echo json_encode(['success' => false, 'message' => 'کد اجباری است.']); return; }
    $stmt = $connect->prepare("UPDATE discount_codes SET code=?, type=?, value=?, max_uses=NULLIF(?, ''), per_user_limit=NULLIF(?, ''), min_order_amount=NULLIF(?, ''), expires_at=NULLIF(?, ''), is_active=?, note=? WHERE id=?");
    $stmt->bind_param('ssdssssisi', $p['code'], $p['type'], $p['value'], $p['max_uses'], $p['per_user_limit'], $p['min_order_amount'], $p['expires_at'], $p['is_active'], $p['note'], $id);
    $ok = $stmt->execute();
    $err = $stmt->error;
    $stmt->close();
    echo json_encode(['success' => $ok, 'message' => $ok ? 'به‌روزرسانی شد' : ('خطا: ' . $err)]);
}

function delete_discount($connect, $id) {
    ensure_discount_schema($connect);
    $id = (int)$id;
    if ($id <= 0) { echo json_encode(['success' => false, 'message' => 'شناسه نامعتبر.']); return; }
    $stmt = $connect->prepare("DELETE FROM discount_codes WHERE id=?");
    $stmt->bind_param('i', $id);
    $ok = $stmt->execute();
    $stmt->close();
    echo json_encode(['success' => $ok]);
}

function toggle_discount_status($connect, $id) {
    ensure_discount_schema($connect);
    $id = (int)$id;
    if ($id <= 0) { echo json_encode(['success' => false, 'message' => 'شناسه نامعتبر.']); return; }
    $stmt = $connect->prepare("UPDATE discount_codes SET is_active = 1 - is_active WHERE id=?");
    $stmt->bind_param('i', $id);
    $ok = $stmt->execute();
    $stmt->close();
    echo json_encode(['success' => $ok]);
}

function get_service_traffic_details($service_and_server_data) {
    $used_gb = null;
    $type = $service_and_server_data['server_type'];
    $link = $service_and_server_data['link'];
    $username = $service_and_server_data['username'];
    $password = $service_and_server_data['password'];
    $service_name = $service_and_server_data['name'];

    try {
        if ($type == "marzban") {
            $token = loginPanel($link, $username, $password)['access_token'] ?? null;
            if ($token) {
                $url = rtrim($link, '/') . "/api/user/$service_name";
                $ch = curl_init($url);
                curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
                curl_setopt($ch, CURLOPT_TIMEOUT, 5);
                curl_setopt($ch, CURLOPT_HTTPHEADER, ['Accept: application/json', 'Authorization: Bearer ' . $token]);
                $response = curl_exec($ch);
                curl_close($ch);
                $json = json_decode($response, true);
                if (isset($json['used_traffic'])) {
                    $bytes = $json['used_traffic'];
                    $used_gb = round($bytes / 1073741824, 2);
                }
            }
        } elseif ($type == 'sanaei' && function_exists('getClientTraffic')) {
            login($link, $username, $password);
            $response = getClientTraffic($link, $service_name);
            $json_data = json_decode($response, true);
            if (isset($json_data['obj'])) {
                $json = $json_data['obj'];
                $bytes = $json['up'] + $json['down'];
                $used_gb = round($bytes / 1073741824, 2);
            }
        }
    } catch (Exception $e) {
        error_log("Error fetching traffic for service {$service_and_server_data['id']}: " . $e->getMessage());
        $used_gb = null;
    }

    $total_gb = (float)$service_and_server_data['size'];
    $remaining_gb = ($used_gb !== null) ? round($total_gb - $used_gb, 2) : null;
    if ($remaining_gb !== null && $remaining_gb < 0) {
        $remaining_gb = 0;
    }

    return [
        'used_traffic' => $used_gb,
        'remaining_traffic' => $remaining_gb
    ];
}

function get_all_services($connect, $page = 1, $search = '', $server_filter = '', $status = '', $include_traffic = true) {
    $limit = 10;
    $offset = ($page - 1) * $limit;
    
    $where_clauses = [];
    $params = [];
    $types = '';

    if (!empty($search)) {
        $where_clauses[] = "(s.display LIKE ? OR s.uuid LIKE ? OR s.owner LIKE ?)";
        $search_term = "%{$search}%";
        array_push($params, $search_term, $search_term, $search_term);
        $types .= 'sss';
    }
    if (!empty($server_filter)) {
        $where_clauses[] = "s.svname = ?";
        $params[] = $server_filter;
        $types .= 's';
    }
    if (!empty($status)) {
        $current_time = time();
        if ($status == 'active') {
            $where_clauses[] = "s.expire > ?";
            $params[] = $current_time;
            $types .= 'i';
        } elseif ($status == 'inactive') {
            $where_clauses[] = "s.expire <= ?";
            $params[] = $current_time;
            $types .= 'i';
        }
    }
    
    $where_sql = count($where_clauses) > 0 ? 'WHERE ' . implode(' AND ', $where_clauses) : '';

    // Count total
    $count_sql = "SELECT COUNT(*) as count FROM services s $where_sql";
    $stmt_count = $connect->prepare($count_sql);
    if (!empty($params)) {
        $stmt_count->bind_param($types, ...$params);
    }
    $stmt_count->execute();
    $total_services = $stmt_count->get_result()->fetch_assoc()['count'];
    $stmt_count->close();
    
    $total_pages = ceil($total_services / $limit);

    // Fetch data
    $sql = "SELECT s.*, sv.type as server_type, sv.link, sv.username, sv.password, sv.inbound_copy
            FROM services s
            LEFT JOIN servers sv ON s.svname = sv.name
            $where_sql ORDER BY s.id DESC LIMIT ? OFFSET ?";
    
    $stmt = $connect->prepare($sql);
    $params[] = $limit;
    $params[] = $offset;
    $types .= 'ii';
    $stmt->bind_param($types, ...$params);
    
    $stmt->execute();
    $result = $stmt->get_result();
    
    $services = [];
    $current_time = time();
    while ($row = $result->fetch_assoc()) {
        $row['is_active'] = $row['expire'] > $current_time;
        if ($include_traffic) {
            $traffic_data = get_service_traffic_details($row);
            $row['used_traffic'] = $traffic_data['used_traffic'];
            $row['remaining_traffic'] = $traffic_data['remaining_traffic'];
        } else {
            $row['used_traffic'] = null;
            $row['remaining_traffic'] = null;
        }
        $services[] = $row;
    }
    $stmt->close();
    
    echo json_encode([
        'success' => true,
        'services' => $services,
        'total_pages' => (int)$total_pages,
        'current_page' => (int)$page
    ]);
}

function delete_service($connect, $service_id) {
    $service_id = (int)$service_id;
    
    $stmt = $connect->prepare("SELECT * FROM services WHERE id = ? LIMIT 1");
    $stmt->bind_param("i", $service_id);
    $stmt->execute();
    $service_res = $stmt->get_result();
    
    if ($service_res->num_rows == 0) {
        echo json_encode(['success' => false, 'message' => 'سرویس یافت نشد.']);
        $stmt->close();
        return;
    }
    $service = $service_res->fetch_assoc();
    $stmt->close();

    $stmt_server = $connect->prepare("SELECT * FROM servers WHERE name = ? LIMIT 1");
    $stmt_server->bind_param("s", $service['svname']);
    $stmt_server->execute();
    $server_res = $stmt_server->get_result();

    if ($server_res->num_rows > 0) {
        $server = $server_res->fetch_assoc();
        try {
            if ($server['type'] == 'marzban') {
                $token = loginPanel($server['link'], $server['username'], $server['password'])['access_token'] ?? null;
                if ($token) {
                    deleteUser($service['name'], $token, $server['link']);
                }
            } elseif ($server['type'] == 'sanaei' && function_exists('DeleteClient')) {
                login($server['link'], $server['username'], $server['password']);
                DeleteClient($service['uuid'], $server['inbound_copy'], $server['link']);
            }
        } catch (Exception $e) {
            error_log("Panel API deletion failed for service ID $service_id: " . $e->getMessage());
        }
    }
    $stmt_server->close();

    $stmt_delete = $connect->prepare("DELETE FROM services WHERE id = ?");
    $stmt_delete->bind_param("i", $service_id);
    if ($stmt_delete->execute()) {
        echo json_encode(['success' => true]);
    } else {
        echo json_encode(['success' => false, 'message' => 'خطا در حذف سرویس از دیتابیس.']);
    }
    $stmt_delete->close();
}

function get_all_users($connect, $page = 1, $search = '', $status = '', $server_filter = '') {
    $limit = 10;
    $offset = ($page - 1) * $limit;
    
    $where_clause = '';
    $params = [];
    $types = '';
    if (!empty($search)) {
        $where_clause = "WHERE id LIKE ? OR username LIKE ?";
        $search_term = "%{$search}%";
        array_push($params, $search_term, $search_term);
        $types .= 'ss';
    }

    // Build additional filters using EXISTS subqueries on services
    $filters_sql = '';
    $now = time();
    if (!empty($status)) {
        if ($status === 'active') {
            // users with at least one active service (expire in future)
            $filters_sql .= (empty($where_clause) ? ' WHERE ' : ' AND ') . "EXISTS (SELECT 1 FROM services s WHERE s.owner = u.id AND s.expire > ?)";
            $params[] = $now; $types .= 'i';
        } elseif ($status === 'inactive') {
            // users with zero active service but may have services
            $filters_sql .= (empty($where_clause) ? ' WHERE ' : ' AND ') . "NOT EXISTS (SELECT 1 FROM services s WHERE s.owner = u.id AND s.expire > ?)";
            $params[] = $now; $types .= 'i';
        } elseif ($status === 'none') {
            // users without any service
            $filters_sql .= (empty($where_clause) ? ' WHERE ' : ' AND ') . "NOT EXISTS (SELECT 1 FROM services s WHERE s.owner = u.id)";
        }
    }

    if (!empty($server_filter)) {
        $filters_sql .= (empty($where_clause) && empty($filters_sql) ? ' WHERE ' : ' AND ') . "EXISTS (SELECT 1 FROM services s2 WHERE s2.owner = u.id AND (s2.svname = ? OR s2.location = ?))";
        $params[] = $server_filter; $params[] = $server_filter; $types .= 'ss';
    }

    $where_full = $where_clause . $filters_sql;

    $count_sql = "SELECT COUNT(*) as count FROM user u $where_full";
    $stmt_count = $connect->prepare($count_sql);
    if(!empty($params)) $stmt_count->bind_param($types, ...$params);
    $stmt_count->execute();
    $total_users = $stmt_count->get_result()->fetch_assoc()['count'];
    $stmt_count->close();
    
    $total_pages = ceil($total_users / $limit);
    
    $sql = "SELECT 
                u.id, u.username, u.coin,
                (SELECT COUNT(*) FROM services s WHERE s.owner = u.id) as service_count,
                (SELECT COUNT(*) FROM services s WHERE s.owner = u.id AND s.expire > $now) as active_service_count
            FROM user u $where_full 
            ORDER BY u.id DESC 
            LIMIT ? OFFSET ?";
    
    $params[] = $limit;
    $params[] = $offset;
    $types .= 'ii';

    $stmt = $connect->prepare($sql);
    $stmt->bind_param($types, ...$params);
    $stmt->execute();
    $result = $stmt->get_result();
    
    $users = [];
    while ($row = $result->fetch_assoc()) {
        $users[] = $row;
    }
    $stmt->close();
    
    echo json_encode([
        'success' => true,
        'users' => $users,
        'total_pages' => (int)$total_pages,
        'current_page' => (int)$page
    ]);
}

function get_stats($connect) {
    $total_users = $connect->query("SELECT COUNT(id) as c FROM user")->fetch_assoc()['c'];
    $total_services = $connect->query("SELECT COUNT(id) as c FROM services WHERE expire > " . time())->fetch_assoc()['c'];
    $total_servers = $connect->query("SELECT COUNT(id) as c FROM servers")->fetch_assoc()['c'];
    echo json_encode(['success' => true, 'stats' => ['total_users' => $total_users, 'total_services' => $total_services, 'total_servers' => $total_servers]]);
}

function get_user_details($connect, $identifier) {
    $identifier = trim($identifier);
    $user_info = null;
    $stmt = null;

    if (is_numeric($identifier)) {
        $stmt = $connect->prepare("SELECT * FROM user WHERE id = ? LIMIT 1");
        $stmt->bind_param("s", $identifier);
    } elseif (strpos($identifier, '@') === 0) {
        $username = substr($identifier, 1);
        $stmt = $connect->prepare("SELECT * FROM user WHERE username = ? LIMIT 1");
        $stmt->bind_param("s", $username);
    } else {
        $stmt_owner = $connect->prepare("SELECT owner FROM services WHERE uuid = ? OR name = ? LIMIT 1");
        $stmt_owner->bind_param("ss", $identifier, $identifier);
        $stmt_owner->execute();
        $owner_res = $stmt_owner->get_result();
        if ($owner_res->num_rows > 0) {
            $owner_id = $owner_res->fetch_assoc()['owner'];
            $stmt = $connect->prepare("SELECT * FROM user WHERE id = ? LIMIT 1");
            $stmt->bind_param("s", $owner_id);
        }
        $stmt_owner->close();
    }

    if ($stmt) {
        $stmt->execute();
        $user_res = $stmt->get_result();
        if ($user_res->num_rows > 0) {
            $user_info = $user_res->fetch_assoc();
        }
        $stmt->close();
    }

    if ($user_info) {
        $user_id = $user_info['id'];
        
        $sql = "SELECT s.*, sv.type as server_type, sv.link, sv.username, sv.password, sv.inbound_copy 
                FROM services s 
                LEFT JOIN servers sv ON s.svname = sv.name 
                WHERE s.owner = ?";
        $stmt_services = $connect->prepare($sql);
        $stmt_services->bind_param("s", $user_id);
        $stmt_services->execute();
        $services_res = $stmt_services->get_result();

        $services_detailed = [];
        $current_time = time();

        while ($service = $services_res->fetch_assoc()) {
            $traffic_data = get_service_traffic_details($service);
            $used_gb = $traffic_data['used_traffic'];
            $total_gb = (float)$service['size'];
            
            $progress_percentage = ($total_gb > 0 && $used_gb !== null) ? round(($used_gb / $total_gb) * 100) : 0;
            $is_active = ($service['expire'] > $current_time) && ($progress_percentage < 100);
            $days_remaining = $is_active ? floor(($service['expire'] - $current_time) / 86400) : 0;

            $service['details'] = [
                'used_traffic' => $used_gb,
                'remaining_traffic' => ($used_gb !== null) ? round($total_gb - $used_gb, 2) : null,
                'progress_percentage' => $progress_percentage,
                'is_active' => $is_active,
                'days_remaining' => $days_remaining,
                'expire_date_jalali' => jdate('Y/m/d', $service['expire']),
                'created_at_jalali' => isset($service['created_at']) ? jdate('Y/m/d H:i', strtotime($service['created_at'])) : 'نامشخص'
            ];
            $services_detailed[] = $service;
        }
        $stmt_services->close();
        
        $user_info['services'] = $services_detailed;
        echo json_encode(['success' => true, 'user' => $user_info]);
    } else {
        echo json_encode(['success' => false, 'message' => 'کاربر یافت نشد.']);
    }
}


function update_balance($connect, $user_id, $amount, $type) {
    $amount = (int)$amount;
    if ($amount <= 0) { echo json_encode(['success' => false, 'message' => 'مبلغ نامعتبر است.']); return; }
    
    $operator = ($type === 'add') ? '+' : '-';
    $sql = "UPDATE user SET coin = coin $operator ? WHERE id = ?";
    $stmt = $connect->prepare($sql);
    $stmt->bind_param("is", $amount, $user_id);
    
    if ($stmt->execute()) { echo json_encode(['success' => true]); } else { echo json_encode(['success' => false, 'message' => 'خطا در بروزرسانی موجودی.']); }
    $stmt->close();
}

function get_servers($connect) {
    $servers = fetch_servers_with_metadata($connect);
    echo json_encode(['success' => true, 'servers' => $servers]);
}

function add_server($connect, $data) {
    if (empty($data['name']) || empty($data['display_name']) || empty($data['type']) || empty($data['link'])) { echo json_encode(['success' => false, 'message' => 'فیلدهای اجباری را پر کنید.']); return; }

    $username = trim($data['username'] ?? '');
    $password = trim($data['password'] ?? '');
    $protocols = trim($data['protocols'] ?? '');
    $inbound_copy = trim($data['inbound_copy'] ?? '');
    $example_link = trim($data['example_link'] ?? '');
    $custom_domain = trim($data['custom_domain'] ?? '');
    $region = trim($data['region'] ?? '');
    $max_capacity = isset($data['max_capacity']) ? (int)$data['max_capacity'] : 0;
    $current_load = isset($data['current_load']) ? (int)$data['current_load'] : 0;
    $maintenance_mode = (!empty($data['maintenance_mode']) && $data['maintenance_mode'] !== '0') ? 1 : 0;
    $maintenance_note = trim($data['maintenance_note'] ?? '');

    $status = 'on';
    $is_hidden = 0;
    $is_test_server = 0;

    $stmt = $connect->prepare("INSERT INTO servers (name, display_name, type, link, username, password, protocols, inbound_copy, example_link, custom_domain, status, is_hidden, is_test_server, region, max_capacity, current_load, maintenance_mode, maintenance_note) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)");
    $stmt->bind_param(
        "sssssssssssiisiiis",
        $data['name'],
        $data['display_name'],
        $data['type'],
        $data['link'],
        $username,
        $password,
        $protocols,
        $inbound_copy,
        $example_link,
        $custom_domain,
        $status,
        $is_hidden,
        $is_test_server,
        $region,
        $max_capacity,
        $current_load,
        $maintenance_mode,
        $maintenance_note
    );

    if ($stmt->execute()) {
        $new_id = $connect->insert_id;
        $stmt->close();

        $tag_ids = [];
        if (!empty($data['tag_ids'])) {
            $tag_ids = array_filter(array_map('intval', explode(',', $data['tag_ids'])));
        }
        if (!empty($tag_ids)) {
            sync_server_tags($connect, $new_id, $tag_ids);
        }

        echo json_encode(['success' => true]);
    } else {
        $error = $stmt->error;
        $stmt->close();
        echo json_encode(['success' => false, 'message' => 'خطا در افزودن سرور: ' . $error]);
    }
}

function update_server($connect, $data) {
    $server_id = isset($data['server_id']) ? (int)$data['server_id'] : 0;
    if ($server_id <= 0) { echo json_encode(['success' => false, 'message' => 'شناسه سرور نامعتبر است.']); return; }
    if (empty($data['name']) || empty($data['display_name']) || empty($data['type']) || empty($data['link'])) { echo json_encode(['success' => false, 'message' => 'فیلدهای اجباری را پر کنید.']); return; }
    $username = trim($data['username'] ?? '');
    $password = trim($data['password'] ?? '');
    $protocols = trim($data['protocols'] ?? '');
    $inbound_copy = trim($data['inbound_copy'] ?? '');
    $example_link = trim($data['example_link'] ?? '');
    $custom_domain = trim($data['custom_domain'] ?? '');
    $region = trim($data['region'] ?? '');
    $max_capacity = isset($data['max_capacity']) ? (int)$data['max_capacity'] : 0;
    $current_load = isset($data['current_load']) ? (int)$data['current_load'] : 0;
    $maintenance_mode = (!empty($data['maintenance_mode']) && $data['maintenance_mode'] !== '0') ? 1 : 0;
    $maintenance_note = trim($data['maintenance_note'] ?? '');

    $stmt = $connect->prepare("UPDATE servers SET name=?, display_name=?, type=?, link=?, username=?, password=?, protocols=?, inbound_copy=?, example_link=?, custom_domain=?, region=?, max_capacity=?, current_load=?, maintenance_mode=?, maintenance_note=? WHERE id=?");
    $stmt->bind_param(
        "sssssssssssiiisi",
        $data['name'],
        $data['display_name'],
        $data['type'],
        $data['link'],
        $username,
        $password,
        $protocols,
        $inbound_copy,
        $example_link,
        $custom_domain,
        $region,
        $max_capacity,
        $current_load,
        $maintenance_mode,
        $maintenance_note,
        $server_id
    );

    if ($stmt->execute()) {
        $stmt->close();
        $tag_ids = [];
        if (!empty($data['tag_ids'])) {
            $tag_ids = array_filter(array_map('intval', explode(',', $data['tag_ids'])));
        }
        sync_server_tags($connect, $server_id, $tag_ids);
        echo json_encode(['success' => true]);
    } else {
        $error = $stmt->error;
        $stmt->close();
        echo json_encode(['success' => false, 'message' => 'خطا در بروزرسانی سرور: ' . $error]);
    }
}

function toggle_server_status($connect, $server_id) {
    $server_id = (int)$server_id;
    
    $stmt = $connect->prepare("SELECT status FROM servers WHERE id=?");
    $stmt->bind_param("i", $server_id);
    $stmt->execute();
    $current_status = $stmt->get_result()->fetch_assoc()['status'];
    $stmt->close();

    $new_status = ($current_status == 'on') ? 'off' : 'on';
    
    $stmt_update = $connect->prepare("UPDATE servers SET status = ? WHERE id = ?");
    $stmt_update->bind_param("si", $new_status, $server_id);
    if ($stmt_update->execute()) { echo json_encode(['success' => true]); } else { echo json_encode(['success' => false, 'message' => 'خطا در تغییر وضعیت.']); }
    $stmt_update->close();
}

function delete_server($connect, $server_id) {
    $server_id = (int)$server_id;
    $stmt = $connect->prepare("DELETE FROM servers WHERE id = ?");
    $stmt->bind_param("i", $server_id);
    if ($stmt->execute()) { echo json_encode(['success' => true]); } else { echo json_encode(['success' => false, 'message' => 'خطا در حذف سرور.']); }
    $stmt->close();
}

function get_plans($connect) {
    $result = $connect->query("SELECT * FROM plans ORDER BY id ASC");
    $plans = [];
    while ($row = $result->fetch_assoc()) { $plans[] = $row; }
    echo json_encode(['success' => true, 'plans' => $plans]);
}

function add_plan($connect, $data) {
    $title = $data['title'];
    $volume = (int)$data['volume'];
    $duration = (int)$data['duration'];
    if (empty($title) || $volume <= 0 || $duration <= 0) { echo json_encode(['success' => false, 'message' => 'اطلاعات نامعتبر است.']); return; }
    $stmt = $connect->prepare("INSERT INTO plans (title, volume, duration) VALUES (?, ?, ?)");
    $stmt->bind_param("sii", $title, $volume, $duration);
    if ($stmt->execute()) { echo json_encode(['success' => true]); } else { echo json_encode(['success' => false, 'message' => 'خطا در افزودن تعرفه.']); }
    $stmt->close();
}

function delete_plan($connect, $plan_id) {
    $plan_id = (int)$plan_id;
    $stmt = $connect->prepare("DELETE FROM plans WHERE id = ?");
    $stmt->bind_param("i", $plan_id);
    if ($stmt->execute()) {
        $stmt_prices = $connect->prepare("DELETE FROM plan_prices WHERE plan_id = ?");
        $stmt_prices->bind_param("i", $plan_id);
        $stmt_prices->execute();
        $stmt_prices->close();
        echo json_encode(['success' => true]);
    } else {
        echo json_encode(['success' => false, 'message' => 'خطا در حذف تعرفه.']);
    }
    $stmt->close();
}

function get_prices($connect, $server_id) {
    $server_id = (int)$server_id;
    $sql = "SELECT p.id as plan_id, p.title, p.volume, p.duration, pp.price, pp.stars_price FROM plans p LEFT JOIN plan_prices pp ON p.id = pp.plan_id AND pp.server_id = ? ORDER BY p.id ASC";
    $stmt = $connect->prepare($sql);
    $stmt->bind_param("i", $server_id);
    $stmt->execute();
    $result = $stmt->get_result();
    $prices = [];
    while ($row = $result->fetch_assoc()) { $prices[] = $row; }
    $stmt->close();
    echo json_encode(['success' => true, 'prices' => $prices]);
}

function set_price($connect, $data) {
    $server_id = (int)$data['server_id'];
    $plan_id = (int)$data['plan_id'];
    $price = (int)$data['price'];
    $stars_price = isset($data['stars_price']) ? (int)$data['stars_price'] : 0;
    if ($price < 0 || $stars_price < 0) { echo json_encode(['success' => false, 'message' => 'قیمت نامعتبر است.']); return; }
    $stmt = $connect->prepare("INSERT INTO plan_prices (plan_id, server_id, price, stars_price) VALUES (?, ?, ?, ?) ON DUPLICATE KEY UPDATE price = VALUES(price), stars_price = VALUES(stars_price)");
    $stmt->bind_param("iiii", $plan_id, $server_id, $price, $stars_price);
    if ($stmt->execute()) { echo json_encode(['success' => true]); } else { echo json_encode(['success' => false, 'message' => 'خطا در ثبت قیمت.']); }
    $stmt->close();
}

function get_messages($connect, $search = '') {
    $where_clause = '';
    $params = [];
    $types = '';
    if (!empty($search)) {
        $where_clause = "WHERE message_key LIKE ? OR message_value LIKE ? OR description LIKE ?";
        $search_term = "%{$search}%";
        array_push($params, $search_term, $search_term, $search_term);
        $types .= 'sss';
    }
    
    $sql = "SELECT * FROM messages $where_clause ORDER BY message_key ASC";
    $stmt = $connect->prepare($sql);
    if(!empty($params)) $stmt->bind_param($types, ...$params);
    $stmt->execute();
    $result = $stmt->get_result();
    
    $messages = [];
    while ($row = $result->fetch_assoc()) {
        $messages[] = $row;
    }
    $stmt->close();
    echo json_encode(['success' => true, 'messages' => $messages]);
}

function update_message($connect, $key, $value) {
    $stmt = $connect->prepare("UPDATE messages SET message_value = ? WHERE message_key = ?");
    $stmt->bind_param("ss", $value, $key);

    if ($stmt->execute()) {
        echo json_encode(['success' => true]);
    } else {
        echo json_encode(['success' => false, 'message' => 'خطا در بروزرسانی پیام.']);
    }
    $stmt->close();
}

function export_users_csv($connect) {
    $now = time();
    $sql = "SELECT u.id, u.name, u.username, u.phone, u.coin,
                   SUM(CASE WHEN s.expire > {$now} THEN 1 ELSE 0 END) AS active_services,
                   COUNT(s.id) AS total_services,
                   COALESCE(o.total_orders, 0) AS total_orders,
                   COALESCE(o.total_paid, 0) AS total_paid
            FROM user u
            LEFT JOIN services s ON s.owner = u.id
            LEFT JOIN (
                SELECT user_id, COUNT(*) AS total_orders, COALESCE(SUM(amount),0) AS total_paid
                FROM orders WHERE status='approved'
                GROUP BY user_id
            ) o ON o.user_id = u.id
            GROUP BY u.id
            ORDER BY u.id ASC";
    $result = $connect->query($sql);
    if (!$result) {
        echo json_encode(['success' => false, 'message' => 'خطا در تولید گزارش کاربران.']);
        return;
    }

    $handle = fopen('php://temp', 'w+');
    if (!$handle) {
        echo json_encode(['success' => false, 'message' => 'خطای داخلی هنگام ساخت فایل.']);
        return;
    }

    fwrite($handle, "\xEF\xBB\xBF");
    fputcsv($handle, ['user_id', 'name', 'username', 'phone', 'balance', 'active_services', 'total_services', 'total_orders', 'total_paid'], ',');

    while ($row = $result->fetch_assoc()) {
        fputcsv($handle, [
            $row['id'],
            $row['name'] ?? '',
            $row['username'] ?? '',
            $row['phone'] ?? '',
            $row['coin'] ?? 0,
            $row['active_services'] ?? 0,
            $row['total_services'] ?? 0,
            $row['total_orders'] ?? 0,
            $row['total_paid'] ?? 0,
        ], ',');
    }

    $result->close();
    rewind($handle);
    $csv = stream_get_contents($handle);
    fclose($handle);

    $filename = 'users-' . date('Ymd-His') . '.csv';
    echo json_encode([
        'success' => true,
        'filename' => $filename,
        'content' => base64_encode($csv),
    ]);
}

function get_extended_stats_api($connect) {
    $stats = compute_extended_stats($connect);
    echo json_encode(['success' => true, 'stats' => $stats]);
}

function get_order_timeline_api($connect, $limit = 15) {
    $orders = fetch_order_timeline($connect, $limit);
    echo json_encode(['success' => true, 'orders' => $orders]);
}

function get_server_tags_api($connect) {
    $tags = fetch_server_tags($connect);
    echo json_encode(['success' => true, 'tags' => $tags]);
}

function create_server_tag($connect, $name, $color) {
    $name = trim($name);
    if (empty($name)) {
        echo json_encode(['success' => false, 'message' => 'نام برچسب الزامی است.']);
        return;
    }

    $color = trim($color);
    if (!preg_match('/^#([0-9a-f]{3}|[0-9a-f]{6})$/i', $color)) {
        $color = '#6366f1';
    }

    $stmt = $connect->prepare("INSERT INTO server_tags (name, color) VALUES (?, ?)");
    $stmt->bind_param('ss', $name, $color);
    if ($stmt->execute()) {
        echo json_encode(['success' => true]);
    } else {
        echo json_encode(['success' => false, 'message' => 'خطا در ثبت برچسب جدید.']);
    }
    $stmt->close();
}

function delete_server_tag($connect, $tag_id) {
    $tag_id = (int)$tag_id;
    if ($tag_id <= 0) {
        echo json_encode(['success' => false, 'message' => 'شناسه برچسب نامعتبر است.']);
        return;
    }

    $stmt = $connect->prepare("DELETE FROM server_tags WHERE id = ?");
    $stmt->bind_param('i', $tag_id);
    if ($stmt->execute()) {
        echo json_encode(['success' => true]);
    } else {
        echo json_encode(['success' => false, 'message' => 'خطا در حذف برچسب.']);
    }
    $stmt->close();
}
?>
