// Admin panel frontend bootstrap (clean version)
document.addEventListener('DOMContentLoaded', () => {
  // --- Elements
  const sidebarLinks = document.querySelectorAll('.sidebar-link');
  const pages = document.querySelectorAll('.page-content');
  const themeToggle = document.getElementById('theme-toggle');
  if (themeToggle) {
    themeToggle.addEventListener('click', () => {
      const isLight = document.documentElement.classList.toggle('light');
      localStorage.setItem('theme', isLight ? 'light' : 'dark');
      const icon = themeToggle.querySelector('i');
      if (icon) {
        icon.classList.toggle('fa-sun', !isLight);
        icon.classList.toggle('fa-moon', isLight);
      }
    });
  }

  // Drawer
  const userDrawer = document.getElementById('user-drawer');
  const userDrawerPanel = document.getElementById('user-drawer-panel');
  const userDrawerOverlay = document.getElementById('user-drawer-overlay');
  const userDrawerClose = document.getElementById('user-drawer-close');
  // Modal (re-implement basic modal used by admin.php)
  const modal = document.getElementById('modal');
  const modalContent = document.getElementById('modal-content');
  // Mobile sidebar toggling
  const sidebarEl = document.getElementById('sidebar');
  const sidebarOverlayEl = document.getElementById('sidebar-overlay');
  const mobileBtn = document.getElementById('mobile-menu-btn');
  function openSidebar(){ sidebarEl?.classList.remove('translate-x-full'); sidebarOverlayEl?.classList.remove('hidden'); }
  function closeSidebar(){ sidebarEl?.classList.add('translate-x-full'); sidebarOverlayEl?.classList.add('hidden'); }
  mobileBtn?.addEventListener('click', openSidebar);
  sidebarOverlayEl?.addEventListener('click', closeSidebar);
  window.addEventListener('resize', () => { if (window.innerWidth >= 768) sidebarOverlayEl?.classList.add('hidden'); });

  // --- Helpers
  window.openUserDrawer = () => {
    if (!userDrawer || !userDrawerPanel) return;
    userDrawer.classList.remove('hidden');
    requestAnimationFrame(() => userDrawerPanel.classList.remove('-translate-x-full'));
  };
  window.closeUserDrawer = () => {
    if (!userDrawer || !userDrawerPanel) return;
    userDrawerPanel.classList.add('-translate-x-full');
    setTimeout(() => userDrawer.classList.add('hidden'), 300);
  };
  userDrawerOverlay?.addEventListener('click', window.closeUserDrawer);
  userDrawerClose?.addEventListener('click', window.closeUserDrawer);
  document.addEventListener('keydown', (e) => { if (e.key === 'Escape') window.closeUserDrawer(); });
  window.openModal = (html) => {
    if (!modal || !modalContent) return;
    modalContent.innerHTML = html || '';
    modal.classList.remove('hidden');
    requestAnimationFrame(() => {
      modal.classList.remove('opacity-0');
      modal.classList.remove('-translate-y-10');
      modalContent.classList.remove('scale-95');
    });
  };
  window.closeModal = () => {
    if (!modal || !modalContent) return;
    modal.classList.add('opacity-0');
    modal.classList.add('-translate-y-10');
    modalContent.classList.add('scale-95');
    setTimeout(() => { modal.classList.add('hidden'); modalContent.innerHTML = ''; }, 250);
  };
  modal?.addEventListener('click', (e) => { if (e.target === modal) window.closeModal(); });

  window.apiRequest = async (action, params = {}) => {
    const body = new URLSearchParams({ action, ...params });
    try {
      const resp = await fetch('api.php', {
        method: 'POST',
        headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
        credentials: 'same-origin',
        cache: 'no-store',
        body
      });
      return await resp.json();
    } catch (e) {
      console.error('API error', e);
      return { success: false, message: 'network_error' };
    }
  };

  window.showLoader = (el) => {
    if (!el) return;
    el.innerHTML = '<div class="flex justify-center p-8"><div class="loader"></div></div>';
  };

  const numberFormatter = new Intl.NumberFormat('fa-IR');
  const currencyFormatter = new Intl.NumberFormat('fa-IR');

  function formatNumber(value) {
    const num = Number(value || 0);
    return numberFormatter.format(isNaN(num) ? 0 : num);
  }

  function formatCurrency(value) {
    return `${formatNumber(value)} تومان`;
  }

  function formatPercent(value) {
    const num = Number(value || 0);
    if (isNaN(num)) return '0%';
    return `${formatNumber(num)}%`;
  }

  function formatDecimal(value, minimumFractionDigits = 0, maximumFractionDigits = 1) {
    const num = Number(value || 0);
    if (isNaN(num)) return '0';
    return new Intl.NumberFormat('fa-IR', { minimumFractionDigits, maximumFractionDigits }).format(num);
  }

  function paymentTypeLabel(type) {
    const key = String(type || '').toLowerCase();
    if (key === 'card' || key === 'card2card') return 'کارت به کارت';
    if (key === 'crypto' || key === 'usdt') return 'رمزارز';
    if (key === 'stars' || key === 'telegram') return 'استارز تلگرام';
    if (key === 'free') return 'رایگان';
    if (key === 'unknown') return 'نامشخص';
    return type || 'نامشخص';
  }

  function buildTagOptions(tags = [], selected = []) {
    if (!tags.length) return '<span class="text-xs text-slate-400">برچسبی ثبت نشده است.</span>';
    const selectedSet = new Set(selected.map(String));
    return tags.map(tag => {
      const checked = selectedSet.has(String(tag.id)) ? 'checked' : '';
      const color = tag.color || '#6366f1';
      return `<label class="tag-choice"><input type="checkbox" name="tag_ids[]" value="${tag.id}" ${checked}><span style="--tag-color:${color}">${tag.name}</span></label>`;
    }).join('');
  }

  // Lightweight cache to speed up page loads
  const cache = { servers: null, plans: null, serverTags: null };

  function renderPagination(containerId, currentPage, totalPages, callback) {
    const container = document.getElementById(containerId);
    if (!container) return;
    if (totalPages <= 1) { container.innerHTML = ''; return; }

    const pagesToShow = 5;
    let startPage = Math.max(1, currentPage - Math.floor(pagesToShow / 2));
    let endPage = Math.min(totalPages, startPage + pagesToShow - 1);
    if (endPage - startPage + 1 < pagesToShow) startPage = Math.max(1, endPage - pagesToShow + 1);

    let html = '<div class="flex items-center justify-center space-x-2 space-x-reverse">';
    html += `<button data-page="${currentPage - 1}" class="pagination-btn" ${currentPage === 1 ? 'disabled' : ''}><i class="fas fa-chevron-right"></i></button>`;
    if (startPage > 1) {
      html += `<button data-page="1" class="pagination-btn">1</button>`;
      if (startPage > 2) html += `<span class="pagination-ellipsis">...</span>`;
    }
    for (let i = startPage; i <= endPage; i++) {
      html += `<button data-page="${i}" class="pagination-btn ${i === currentPage ? 'active' : ''}">${i}</button>`;
    }
    if (endPage < totalPages) {
      if (endPage < totalPages - 1) html += `<span class="pagination-ellipsis">...</span>`;
      html += `<button data-page="${totalPages}" class="pagination-btn">${totalPages}</button>`;
    }
    html += `<button data-page="${currentPage + 1}" class="pagination-btn" ${currentPage === totalPages ? 'disabled' : ''}><i class="fas fa-chevron-left"></i></button></div>`;
    container.innerHTML = html;
    container.querySelectorAll('.pagination-btn:not([disabled])').forEach(btn => btn.addEventListener('click', () => callback(parseInt(btn.dataset.page))));
  }

  async function fetchServerTags(force = false) {
    if (!force && cache.serverTags) return cache.serverTags;
    const res = await apiRequest('get_server_tags');
    cache.serverTags = (res.success && Array.isArray(res.tags)) ? res.tags : [];
    return cache.serverTags;
  }

  async function renderServerTagsManager(force = false) {
    const card = document.getElementById('server-tags-card');
    const list = document.getElementById('server-tags-list');
    if (!card || !list) return;
    showLoader(list);
    const tags = await fetchServerTags(force);
    if (!tags.length) {
      list.innerHTML = '<p class="text-xs text-slate-400">برچسبی ثبت نشده است. از فرم بالا برای افزودن استفاده کنید.</p>';
    } else {
      list.innerHTML = tags.map(tag => `
        <div class="flex items-center justify-between rounded-xl bg-slate-800/30 px-4 py-3">
          <div>
            <span class="tag-pill" style="--tag-color:${tag.color || '#6366f1'}">${tag.name}</span>
            <span class="text-xs text-slate-400 mr-3">${formatNumber(tag.usage_count)} سرور</span>
          </div>
          <button class="btn-icon text-red-400" data-delete-tag="${tag.id}"><i class="fas fa-trash"></i></button>
        </div>`).join('');
      list.querySelectorAll('[data-delete-tag]')?.forEach(btn => btn.addEventListener('click', async () => {
        const tagId = btn.getAttribute('data-delete-tag');
        if (!confirm('این برچسب حذف شود؟')) return;
        const res = await apiRequest('delete_server_tag', { tag_id: tagId });
        if (res.success) {
          cache.serverTags = null;
          await renderServerTagsManager(true);
          loadServers();
        } else {
          alert(res.message || 'خطا در حذف برچسب');
        }
      }));
    }
  }

  function renderServerHealthOverview(servers = []) {
    const container = document.getElementById('server-health-overview');
    if (!container) return;
    if (!servers.length) {
      container.innerHTML = '<p class="text-sm text-slate-400">سروری ثبت نشده است.</p>';
      return;
    }
    let online = 0, maintenance = 0, offline = 0;
    let totalCapacity = 0, totalLoad = 0;
    servers.forEach(server => {
      const status = server.health_status || '';
      if (status === 'maintenance') maintenance++;
      else if (status === 'online' || status === 'warning') online++;
      else offline++;
      totalCapacity += Number(server.max_capacity || 0);
      totalLoad += Number(server.current_load || 0);
    });
    const avgUtil = servers.filter(s => s.utilization !== null && s.utilization !== undefined);
    const avgUtilValue = avgUtil.length ? Math.round(avgUtil.reduce((sum, s) => sum + Number(s.utilization), 0) / avgUtil.length) : null;
    const sorted = [...servers].sort((a, b) => Number(b.current_load || 0) - Number(a.current_load || 0)).slice(0, 5);
    container.innerHTML = `
      <div class="grid grid-cols-1 sm:grid-cols-2 gap-3">
        <div class="analytics-chip bg-emerald-500/10 text-emerald-300"><i class="fas fa-signal ml-2"></i><span>آنلاین: ${formatNumber(online)}</span></div>
        <div class="analytics-chip bg-amber-500/10 text-amber-300"><i class="fas fa-screwdriver-wrench ml-2"></i><span>نگهداری: ${formatNumber(maintenance)}</span></div>
        <div class="analytics-chip bg-rose-500/10 text-rose-300"><i class="fas fa-power-off ml-2"></i><span>خاموش: ${formatNumber(offline)}</span></div>
        <div class="analytics-chip bg-sky-500/10 text-sky-300"><i class="fas fa-database ml-2"></i><span>ظرفیت مصرفی: ${formatNumber(totalLoad)} / ${formatNumber(totalCapacity)}</span></div>
      </div>
      <div class="mt-4 space-y-3">
        ${avgUtilValue !== null ? `<div class="text-sm text-slate-300">میانگین استفاده: <span class="font-semibold">${formatPercent(avgUtilValue)}</span></div>` : ''}
        <div class="space-y-2">
          ${sorted.map(server => {
            const util = server.utilization !== null ? Math.min(100, Number(server.utilization)) : null;
            return `<div>
              <div class="flex items-center justify-between text-xs text-slate-300 mb-1"><span>${server.display_name || server.name}</span><span>${util !== null ? formatPercent(util) : '---'}</span></div>
              <div class="progress"><span style="width:${util !== null ? util : 0}%"></span></div>
            </div>`;
          }).join('')}
        </div>
      </div>`;
  }

  function renderFinancialOverview(stats) {
    const container = document.getElementById('financial-overview');
    if (!container) return;
    if (!stats) {
      container.innerHTML = '<div class="text-sm text-slate-400">اطلاعات مالی در دسترس نیست.</div>';
      return;
    }
    const paymentBreakdown = Array.isArray(stats.payment_breakdown) ? stats.payment_breakdown : [];
    const totalPayment = paymentBreakdown.reduce((sum, item) => sum + Number(item.total_amount || 0), 0);
    const retention = stats.retention || {};
    const repeatRate = Number(retention.repeat_rate || 0);
    const repeatRateWidth = Math.max(0, Math.min(100, repeatRate));
    const avgOrders = Number(retention.average_orders_per_customer || 0);
    const topRegions = Array.isArray(stats.region_breakdown) ? stats.region_breakdown.slice(0, 4) : [];

    container.innerHTML = `
      <div class="analytics-card">
        <div class="text-xs text-slate-400 mb-2">درآمد ۳۰ روز اخیر</div>
        <div class="text-2xl font-bold">${formatNumber(stats.revenue_30_days)}</div>
        <div class="text-xs text-emerald-300 mt-1">${stats.revenue_growth !== null ? (stats.revenue_growth >= 0 ? '🔼' : '🔻') + ' ' + Math.abs(stats.revenue_growth) + '%' : ''}</div>
      </div>
      <div class="analytics-card">
        <div class="text-xs text-slate-400 mb-2">میانگین مبلغ سفارش</div>
        <div class="text-2xl font-bold">${formatNumber(Math.round(stats.avg_order_amount || 0))}</div>
        <div class="text-xs text-slate-400 mt-1">تومان</div>
      </div>
      <div class="analytics-card">
        <div class="text-xs text-slate-400 mb-2">سفارشات در انتظار</div>
        <div class="text-2xl font-bold">${formatNumber(stats.pending_orders)}</div>
        <div class="text-xs text-slate-400 mt-1">در صف تایید</div>
      </div>
      <div class="analytics-card">
        <div class="text-xs text-slate-400 mb-2">کاربران جدید ۷ روز اخیر</div>
        <div class="text-2xl font-bold">${formatNumber(stats.new_users_7_days)}</div>
        <div class="text-xs text-slate-400 mt-1">عضو تازه</div>
      </div>
      <div class="analytics-card xl:col-span-2">
        <div class="flex items-center justify-between mb-3">
          <div class="text-xs text-slate-400">تفکیک روش‌های پرداخت</div>
          <div class="text-xs text-slate-300">مجموع: ${formatCurrency(totalPayment)}</div>
        </div>
        ${paymentBreakdown.length ? `
          <table class="mini-table insight-table">
            <thead><tr><th>روش</th><th>سفارش</th><th>مبلغ</th><th>سهم</th></tr></thead>
            <tbody>
              ${paymentBreakdown.map(item => {
                const amount = Number(item.total_amount || 0);
                const share = totalPayment > 0 ? Math.round((amount / totalPayment) * 100) : 0;
                return `<tr>
                  <td>${paymentTypeLabel(item.type)}</td>
                  <td>${formatNumber(item.orders_count)}</td>
                  <td>${formatCurrency(amount)}</td>
                  <td>${formatPercent(share)}</td>
                </tr>`;
              }).join('')}
            </tbody>
          </table>
        ` : '<p class="text-xs text-slate-400">داده‌ای برای روش‌های پرداخت موجود نیست.</p>'}
      </div>
      <div class="analytics-card xl:col-span-2 retention-card">
        <div class="flex items-center justify-between gap-3">
          <div>
            <div class="text-xs text-slate-400">وفاداری مشتریان (${retention.period_days || 30} روز اخیر)</div>
            <div class="text-2xl font-bold mt-1">${formatDecimal(repeatRate, 0, 1)}%</div>
          </div>
          <div class="retention-badge">میانگین سفارش / کاربر: ${formatDecimal(avgOrders, 0, 2)}</div>
        </div>
        <div class="retention-meter mt-4">
          <div class="progress"><span style="width:${repeatRateWidth}%"></span></div>
          <div class="text-xs text-slate-400 mt-2">نرخ بازگشت مشتریان فعال</div>
        </div>
        <div class="retention-grid mt-4">
          <div class="retention-chip">
            <span class="label">کاربران فعال</span>
            <span class="value">${formatNumber(retention.active_customers || 0)}</span>
          </div>
          <div class="retention-chip">
            <span class="label">بازگشتی</span>
            <span class="value">${formatNumber(retention.returning_customers || 0)}</span>
          </div>
          <div class="retention-chip">
            <span class="label">جدید</span>
            <span class="value">${formatNumber(retention.new_customers || 0)}</span>
          </div>
          ${retention.dormant_customers ? `<div class="retention-chip">
            <span class="label">غیرفعال</span>
            <span class="value">${formatNumber(retention.dormant_customers || 0)}</span>
          </div>` : ''}
        </div>
      </div>
      <div class="analytics-card xl:col-span-2 region-card">
        <div class="flex items-center justify-between mb-3">
          <div class="text-xs text-slate-400">پوشش منطقه‌ای سرورها</div>
          <button class="liquid-pill px-3 py-1 text-xs" onclick="document.querySelector('[data-target=servers]')?.click()">مدیریت سرورها</button>
        </div>
        ${topRegions.length ? `
          <ul class="insight-list">
            ${topRegions.map(region => {
              const util = region.utilization !== null ? Math.min(100, Number(region.utilization)) : null;
              const capacityText = `${formatNumber(region.total_load || 0)} / ${formatNumber(region.total_capacity || 0)}`;
              const maintenanceText = region.maintenance_count ? ` | نگهداری: ${formatNumber(region.maintenance_count)}` : '';
              const availableText = region.available_capacity ? ` | آزاد: ${formatNumber(region.available_capacity)}` : '';
              return `<li class="insight-row">
                <div>
                  <div class="insight-label">${region.region}</div>
                  <div class="insight-sub">سرور: ${formatNumber(region.server_count || 0)} | آنلاین: ${formatNumber(region.online_count || 0)} | بار: ${capacityText}${maintenanceText}${availableText}</div>
                </div>
                <div class="insight-progress">
                  <span>${util !== null ? formatPercent(util) : '---'}</span>
                  <div class="progress"><span style="width:${util !== null ? util : 0}%"></span></div>
                </div>
              </li>`;
            }).join('')}
          </ul>
        ` : '<p class="text-xs text-slate-400">هیچ منطقه‌ای ثبت نشده است.</p>'}
      </div>`;
  }

  function renderRevenueTrend(trend = []) {
    const container = document.getElementById('revenue-trend');
    if (!container) return;
    if (!trend.length) {
      container.innerHTML = '<p class="text-sm text-slate-400">داده‌ای برای نمایش موجود نیست.</p>';
      return;
    }
    container.innerHTML = `<table class="mini-table"><thead><tr><th>تاریخ</th><th>سفارش</th><th>مبلغ</th></tr></thead><tbody>
      ${trend.map(item => `<tr><td>${item.day}</td><td>${formatNumber(item.orders_count)}</td><td>${formatNumber(item.total_amount)}</td></tr>`).join('')}
    </tbody></table>`;
  }

  async function renderOrderTimeline(force = false) {
    const container = document.getElementById('order-timeline');
    if (!container) return;
    showLoader(container);
    const data = await apiRequest('get_order_timeline');
    if (!data.success || !Array.isArray(data.orders) || !data.orders.length) {
      container.innerHTML = '<p class="text-sm text-slate-400">سفارشی ثبت نشده است.</p>';
      return;
    }
    const statusClass = (status) => {
      const key = String(status || '').toLowerCase();
      if (key === 'approved') return 'badge badge-online';
      if (key === 'pending') return 'badge badge-warn';
      if (key === 'rejected') return 'badge badge-danger';
      return 'badge badge-neutral';
    };
    const statusLabel = (status) => {
      const key = String(status || '').toLowerCase();
      if (key === 'approved') return 'تایید شده';
      if (key === 'pending') return 'در انتظار';
      if (key === 'rejected') return 'رد شده';
      return status || 'نامشخص';
    };
    container.innerHTML = '<ul class="timeline">' + data.orders.map(order => `
      <li>
        <div class="flex items-center justify-between mb-1">
          <span class="font-medium">#${order.order_code}</span>
          <span class="text-xs text-slate-400">${order.created_at_jalali || ''}</span>
        </div>
        <div class="text-xs text-slate-300 mb-1">${paymentTypeLabel(order.type)} · ${order.location || '-'}</div>
        <div class="flex items-center justify-between">
          <span class="text-sm font-semibold">${formatNumber(order.amount)} تومان</span>
          <span class="${statusClass(order.status)}">${statusLabel(order.status)}</span>
        </div>
      </li>`).join('') + '</ul>';
  }

  const exportBtn = document.getElementById('users-export-btn');
  if (exportBtn && !exportBtn.dataset.wired) {
    exportBtn.dataset.wired = '1';
    exportBtn.addEventListener('click', async () => {
      const original = exportBtn.innerHTML;
      exportBtn.disabled = true;
      exportBtn.innerHTML = '<i class="fas fa-spinner fa-spin"></i><span> در حال آماده‌سازی...</span>';
      const res = await apiRequest('export_users_csv');
      if (res.success && res.content) {
        const link = document.createElement('a');
        link.href = `data:text/csv;base64,${res.content}`;
        link.download = res.filename || 'users.csv';
        document.body.appendChild(link);
        link.click();
        document.body.removeChild(link);
      } else {
        alert(res.message || 'خطا در تهیه خروجی کاربران.');
      }
      exportBtn.disabled = false;
      exportBtn.innerHTML = original;
    });
  }

  const tagForm = document.getElementById('server-tag-form');
  if (tagForm && !tagForm.dataset.wired) {
    tagForm.dataset.wired = '1';
    tagForm.addEventListener('submit', async (e) => {
      e.preventDefault();
      const formData = new FormData(tagForm);
      const name = (formData.get('name') || '').toString().trim();
      const color = (formData.get('color') || '').toString().trim();
      if (!name) {
        alert('نام برچسب را وارد کنید.');
        return;
      }
      const submitBtn = tagForm.querySelector('button[type="submit"]');
      const original = submitBtn ? submitBtn.innerHTML : '';
      if (submitBtn) { submitBtn.disabled = true; submitBtn.innerHTML = '<i class="fas fa-spinner fa-spin"></i>'; }
      const res = await apiRequest('create_server_tag', { name, color });
      if (submitBtn) { submitBtn.disabled = false; submitBtn.innerHTML = original || 'افزودن برچسب'; }
      if (res.success) {
        tagForm.reset();
        cache.serverTags = null;
        await renderServerTagsManager(true);
        loadServers();
      } else {
        alert(res.message || 'خطا در ثبت برچسب');
      }
    });
  }

  const refreshTagsBtn = document.getElementById('refresh-server-tags');
  if (refreshTagsBtn && !refreshTagsBtn.dataset.wired) {
    refreshTagsBtn.dataset.wired = '1';
    refreshTagsBtn.addEventListener('click', async () => {
      refreshTagsBtn.classList.add('opacity-70');
      cache.serverTags = null;
      await renderServerTagsManager(true);
      refreshTagsBtn.classList.remove('opacity-70');
    });
  }

  // --- Page loaders
  async function loadDashboard() {
    const statsGrid = document.getElementById('stats-grid');
    showLoader(statsGrid);
    const data = await apiRequest('get_stats');
    if (!data.success) {
      statsGrid.innerHTML = '<div class="col-span-full text-center p-4">خطا در بارگذاری آمار</div>';
      return;
    }
    statsGrid.innerHTML = `
      <div class="bg-white p-5 md:p-6 flex items-center justify-between stat-kpi">
        <div class="mr-4"><h3 class="font-medium">کل کاربران</h3><p class="text-2xl md:text-3xl font-bold">${data.stats.total_users}</p></div>
        <div class="icon-pill"><i class="fas fa-users"></i></div>
      </div>
      <div class="bg-white p-5 md:p-6 flex items-center justify-between stat-kpi">
        <div class="mr-4"><h3 class="font-medium">سرویس‌های فعال</h3><p class="text-2xl md:text-3xl font-bold">${data.stats.total_services}</p></div>
        <div class="icon-pill"><i class="fas fa-rocket"></i></div>
      </div>
      <div class="bg-white p-5 md:p-6 flex items-center justify-between stat-kpi">
        <div class="mr-4"><h3 class="font-medium">تعداد سرورها</h3><p class="text-2xl md:text-3xl font-bold">${data.stats.total_servers}</p></div>
        <div class="icon-pill"><i class="fas fa-server"></i></div>
      </div>`;

    const extended = await apiRequest('get_extended_stats');
    if (extended.success) {
      renderFinancialOverview(extended.stats);
      renderRevenueTrend(extended.stats?.daily_trend || []);
    } else {
      renderFinancialOverview(null);
      renderRevenueTrend([]);
    }
    await renderOrderTimeline();

    const dash = document.getElementById('dashboard');
    if (!dash) return;
    let widgets = document.getElementById('dashboard-widgets');
    if (!widgets) {
      widgets = document.createElement('div');
      widgets.id = 'dashboard-widgets';
      widgets.className = 'mt-6 grid grid-cols-1 lg:grid-cols-3 gap-6';
      dash.appendChild(widgets);
    }
    widgets.innerHTML = `
      <div class="lg:col-span-2 bg-white p-6">
        <div class="flex items-center justify-between mb-4">
          <h3 class="widget-title font-medium">فعالیت‌های اخیر</h3>
          <button class="liquid-pill px-3 py-1 text-sm" onclick="document.querySelector('[data-target=users-list]')?.click()">مشاهده کاربران</button>
        </div>
        <div id="recent-services"></div>
      </div>
      <div>
        <div class="bg-white p-6">
          <h3 class="widget-title font-medium mb-4">میانبرها</h3>
          <div id="quick-actions" class="grid grid-cols-2 gap-3"></div>
        </div>
        <div class="bg-white p-6 mt-6">
          <h3 class="widget-title font-medium mb-4">وضعیت سرورها</h3>
          <div id="servers-mini"></div>
        </div>
      </div>`;

    const quick = document.getElementById('quick-actions');
    if (quick) {
      quick.innerHTML = `
        <button class="quick-action" onclick="document.querySelector('[data-target=user-search]')?.click()"><i class="fas fa-magnifying-glass ml-2"></i><span>جستجوی کاربر</span></button>
        <button class="quick-action" onclick="document.querySelector('[data-target=users-list]')?.click()"><i class="fas fa-rectangle-list ml-2"></i><span>کاربران</span></button>
        <button class="quick-action" onclick="document.querySelector('[data-target=servers]')?.click()"><i class="fas fa-server ml-2"></i><span>سرورها</span></button>
        <button class="quick-action" onclick="document.querySelector('[data-target=broadcast]')?.click()"><i class="fas fa-bullhorn ml-2"></i><span>پیام همگانی</span></button>`;
    }

    const refreshTimeline = document.getElementById('refresh-orders-timeline');
    if (refreshTimeline && !refreshTimeline.dataset.wired) {
      refreshTimeline.dataset.wired = '1';
      refreshTimeline.addEventListener('click', () => renderOrderTimeline(true));
    }

    const recent = document.getElementById('recent-services');
    if (recent) {
      showLoader(recent);
      const svcData = await apiRequest('get_all_services', { page: 1, include_traffic: false });
      if (svcData.success && Array.isArray(svcData.services) && svcData.services.length) {
        recent.innerHTML = '<ul class="glass-list">' + svcData.services.slice(0, 5).map(s => `
          <li class="glass-row">
            <div class="title">${s.display || '-'}</div>
            <div class="meta">${s.owner || '-'}</div>
            <div class="meta">${s.location || '-'}</div>
            <div class="badge">${s.expire ? new Date(s.expire * 1000).toLocaleDateString('fa-IR') : '-'}</div>
          </li>`).join('') + '</ul>';
      } else {
        recent.innerHTML = '<div class="p-4 text-center">موردی یافت نشد.</div>';
      }
    }

    const serversMini = document.getElementById('servers-mini');
    if (serversMini) {
      if (extended.success) {
        const stats = extended.stats || {};
        const health = stats.server_health || {};
        const topServers = Array.isArray(stats.top_servers) ? stats.top_servers.slice(0, 3) : [];
        serversMini.innerHTML = `
          <div class="space-y-2 text-xs text-slate-300">
            <div>آنلاین: <span class="font-semibold text-emerald-300">${formatNumber(health.online || 0)}</span></div>
            <div>نگهداری: <span class="font-semibold text-amber-300">${formatNumber(health.maintenance || 0)}</span></div>
            <div>خاموش: <span class="font-semibold text-rose-300">${formatNumber(health.offline || 0)}</span></div>
          </div>
          <div class="mt-4">
            <h4 class="text-xs font-semibold text-slate-200 mb-2">پربازده‌ترین سرورها</h4>
            ${topServers.length ? '<ul class="space-y-1 text-xs">' + topServers.map(server => `<li class="flex items-center justify-between"><span>${server.display_name || server.id}</span><span>${formatPercent(server.utilization || 0)}</span></li>`).join('') + '</ul>' : '<p class="text-xs text-slate-400">داده‌ای موجود نیست.</p>'}
          </div>`;
      } else {
        serversMini.innerHTML = '<div class="p-4 text-center">خطا در دریافت وضعیت سرورها.</div>';
      }
    }
  }

  async function loadUsersList(page = 1, search = '', status = '', server = '') {
    const container = document.getElementById('users-table-container');
    showLoader(container);
    const data = await apiRequest('get_all_users', { page, search, status, server });
    if (!data.success) { container.innerHTML = '<p class="text-center p-4">خطا در بارگذاری کاربران.</p>'; return; }
    container.innerHTML = (data.users && data.users.length > 0) ? `
      <div class="overflow-hidden rounded-xl border border-white/10">
        <table class="w-full text-right text-sm table-cool">
          <thead>
            <tr>
              <th>ID کاربر</th>
              <th>نام کاربری</th>
              <th>موجودی (تومان)</th>
              <th>تعداد سرویس</th>
            </tr>
          </thead>
          <tbody>
            ${data.users.map(u => `
              <tr class="cursor-pointer" data-user-id="${u.id}">
                <td class="font-mono text-sm">
                  <code class="user-id-link text-indigo-300 hover:text-indigo-200" data-user-id="${u.id}">${u.id}</code>
                </td>
                <td class="truncate">${u.username ? '@' + u.username : '<span class="opacity-60">—</span>'}</td>
                <td class="font-semibold">${new Intl.NumberFormat('fa-IR').format(u.coin)}</td>
                <td class="text-sm">
                  <span class="font-semibold">${u.service_count}</span>
                  <span class="text-xs text-slate-400">(${u.active_service_count ?? 0} فعال)</span>
                </td>
              </tr>
            `).join('')}
          </tbody>
        </table>
      </div>` : '<p class="text-center p-4">کاربری یافت نشد.</p>';
    renderPagination('users-pagination', data.current_page, data.total_pages, (newPage) => loadUsersList(newPage, search, status, server));

    // User details drawer
    // row click (and ID click) → open modal with details
    const openDetails = async (uid) => {
      openModal(`
          <div class="flex items-center justify-between mb-3">
            <h3 class="text-lg font-bold">جزئیات کاربر</h3>
            <button onclick="closeModal()" class="text-2xl">&times;</button>
          </div>
          <div id="user-details-modal" class="max-h-[70vh] overflow-y-auto"></div>
        `);
      const body = document.getElementById('user-details-modal');
      if (body) showLoader(body);
      const d = await apiRequest('get_user_details', { identifier: uid });
      renderUserDetailsInto('user-details-modal', d);
    };
    container.querySelectorAll('.user-id-link')?.forEach(el => {
      el.addEventListener('click', async (e) => { e.stopPropagation(); const uid = el.getAttribute('data-user-id'); openDetails(uid); });
    });
    container.querySelectorAll('tbody tr')?.forEach(row => {
      row.addEventListener('click', () => { const uid = row.getAttribute('data-user-id'); openDetails(uid); });
    });
  }

  async function loadServicesList(page = 1, search = '', server = '', status = '') {
    const container = document.getElementById('services-table-container');
    showLoader(container);
    const data = await apiRequest('get_all_services', { page, search, server, status, include_traffic: true });
    if (!data.success) { container.innerHTML = '<p class="text-center p-4">خطا در بارگذاری سرویس‌ها.</p>'; return; }
    container.innerHTML = (data.services && data.services.length > 0) ? `
      <table class="w-full text-right text-sm table-cool">
        <thead>
          <tr>
            <th class="p-3">نام سرویس</th>
            <th class="p-3">مالک (ID)</th>
            <th class="p-3">سرور</th>
            <th class="p-3">مصرف / کل (GB)</th>
            <th class="p-3">انقضا</th>
            <th class="p-3">وضعیت</th>
          </tr>
        </thead>
        <tbody>
          ${data.services.map(s => `
            <tr>
              <td class="p-4 font-mono text-xs">${s.display}</td>
              <td class="p-4"><code>${s.owner}</code></td>
              <td class="p-4">${s.location}</td>
              <td class="p-4">${s.used_traffic !== null ? `${s.used_traffic} / ${s.size}` : `- / ${s.size}`}</td>
              <td class="p-4">${new Date(s.expire * 1000).toLocaleDateString('fa-IR')}</td>
              <td class="p-4">${s.is_active ? '<span class="px-2 py-1 bg-green-500/20 text-green-300 rounded-full text-xs font-medium">فعال</span>' : '<span class="px-2 py-1 bg-red-500/20 text-red-300 rounded-full text-xs font-medium">غیرفعال</span>'}</td>
            </tr>`).join('')}
        </tbody>
      </table>` : '<p class="text-center p-4">سرویسی یافت نشد.</p>';
    renderPagination('services-pagination', data.current_page, data.total_pages, (newPage) => loadServicesList(newPage, search, server, status));
  }

  // --- Servers page
  async function loadServers() {
    const list = document.getElementById('server-list');
    if (!list) return;
    showLoader(list);
    const data = await apiRequest('get_servers');
    if (!data.success) { list.innerHTML = '<p class="text-center p-4">خطا در بارگذاری سرورها.</p>'; return; }
    if (!data.servers || data.servers.length === 0) { list.innerHTML = '<p class="text-center p-4">هیچ سروری یافت نشد.</p>'; return; }
    cache.servers = data.servers;
    window.serverMap = {}; data.servers.forEach(s => window.serverMap[s.id] = s);
    list.innerHTML = `
      <table class="w-full text-right table-cool">
        <thead>
          <tr>
            <th class="p-3">سرور</th>
            <th class="p-3">منطقه</th>
            <th class="p-3">ظرفیت</th>
            <th class="p-3">برچسب‌ها</th>
            <th class="p-3">وضعیت</th>
            <th class="p-3">اقدامات</th>
          </tr>
        </thead>
        <tbody>
          ${data.servers.map(server => {
            const utilization = server.utilization !== null ? Math.min(100, Number(server.utilization)) : null;
            const status = server.health_status || (server.status === 'on' ? 'online' : 'offline');
            const statusBadge = status === 'maintenance'
              ? '<span class="badge badge-maintenance">نگهداری</span>'
              : status === 'warning'
                ? '<span class="badge badge-warn">هشدار بار</span>'
                : status === 'online'
                  ? '<span class="badge badge-online">آنلاین</span>'
                  : '<span class="badge badge-danger">خاموش</span>';
            const extraBadge = String(server.is_hidden) === '1' ? '<span class="badge badge-neutral mt-2">مخفی</span>' : '';
            const tagHtml = (Array.isArray(server.tags) && server.tags.length)
              ? server.tags.map(tag => `<span class="tag-pill" style="--tag-color:${tag.color || '#6366f1'}">${tag.name}</span>`).join('')
              : '<span class="text-xs text-slate-400">بدون برچسب</span>';
            const capacityText = server.max_capacity && Number(server.max_capacity) > 0
              ? `${formatNumber(server.current_load)} / ${formatNumber(server.max_capacity)}`
              : `${formatNumber(server.current_load)} / ∞`;
            const progress = utilization !== null ? `<div class="progress mt-2"><span style="width:${utilization}%"></span></div>` : '';
            return `<tr>
              <td class="p-4">
                <div class="font-semibold">${server.display_name || server.name}</div>
                <div class="text-xs text-slate-400 mt-1">${server.type || '-'}</div>
              </td>
              <td class="p-4 text-sm">${server.region || 'نامشخص'}</td>
              <td class="p-4 text-sm">
                <div>${capacityText}</div>
                ${progress}
              </td>
              <td class="p-4"><div class="flex flex-wrap gap-2">${tagHtml}</div></td>
              <td class="p-4 space-y-1">${statusBadge}${extraBadge}</td>
              <td class="p-4 space-x-2 space-x-reverse">
                <button class="btn-icon" data-act="edit" data-id="${server.id}" title="ویرایش"><i class="fas fa-gear"></i></button>
                <button class="btn-icon" data-act="toggle" data-id="${server.id}" title="روشن/خاموش"><i class="fas fa-power-off"></i></button>
                <button class="btn-icon" data-act="hidden" data-id="${server.id}" title="نمایش/مخفی"><i class="fas ${server.is_hidden == 1 ? 'fa-eye-slash' : 'fa-eye'}"></i></button>
                <button class="btn-icon ${server.is_test_server == 1 ? 'text-green-400' : ''}" data-act="test" data-id="${server.id}" title="سرور تست"><i class="fas fa-vial-circle-check"></i></button>
                <button class="btn-icon text-red-400 hover:text-red-500" data-act="delete" data-id="${server.id}" title="حذف"><i class="fas fa-trash"></i></button>
              </td>
            </tr>`;
          }).join('')}
        </tbody>
      </table>`;

    renderServerHealthOverview(data.servers);
    renderServerTagsManager();

    // actions
    list.querySelectorAll('button[data-act]')?.forEach(btn => btn.addEventListener('click', async () => {
      const act = btn.getAttribute('data-act');
      const id = parseInt(btn.getAttribute('data-id'));
      if (act === 'toggle') {
        const r = await apiRequest('toggle_server_status', { server_id: id });
        if (r.success) loadServers(); else alert('خطا در تغییر وضعیت');
      } else if (act === 'hidden') {
        const r = await apiRequest('toggle_server_hidden', { server_id: id });
        if (r.success) loadServers(); else alert('خطا در تغییر وضعیت نمایش');
      } else if (act === 'test') {
        const r = await apiRequest('set_test_server', { server_id: id });
        if (r.success) loadServers(); else alert('خطا در انتخاب سرور تست');
      } else if (act === 'delete') {
        if (!confirm('آیا از حذف این سرور مطمئن هستید؟')) return;
        const r = await apiRequest('delete_server', { server_id: id });
        if (r.success) loadServers(); else alert('خطا در حذف سرور');
      } else if (act === 'edit') {
        openEditServerModal(id);
      }
    }));

    const addBtn = document.getElementById('add-server-btn');
    if (addBtn && !addBtn.dataset.wired) {
      addBtn.dataset.wired = '1';
      addBtn.addEventListener('click', () => openAddServerModal());
    }
  }

  async function openAddServerModal() {
    const tags = await fetchServerTags();
    const html = `
      <h3 class="text-xl font-bold mb-6 border-b pb-3 border-slate-600">افزودن سرور جدید</h3>
      <form id="add-server-form" class="space-y-4 text-sm">
        <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
          <div><label class="font-medium">نام سیستمی:</label><input type="text" name="name" class="w-full p-2 mt-1" required></div>
          <div><label class="font-medium">نام نمایشی:</label><input type="text" name="display_name" class="w-full p-2 mt-1" required></div>
        </div>
        <div>
          <label class="font-medium">نوع سرور:</label>
          <select name="type" class="w-full p-2 bg-slate-700 mt-1" required>
            <option value="marzban">Marzban</option>
            <option value="sanaei">Sanaei</option>
          </select>
        </div>
        <div><label class="font-medium">لینک پنل:</label><input type="text" name="link" class="w-full p-2 mt-1" placeholder="https://host:port" required></div>
        <div><label class="font-medium">دامنه جایگزین اشتراک (اختیاری):</label><input type="text" name="custom_domain" class="w-full p-2 mt-1" placeholder="sub.domain.com"></div>
        <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
          <div><label class="font-medium">یوزرنیم (اختیاری):</label><input type="text" name="username" class="w-full p-2 mt-1" placeholder="admin"></div>
          <div><label class="font-medium">پسورد (اختیاری):</label><input type="text" name="password" class="w-full p-2 mt-1" placeholder="••••••"></div>
        </div>
        <div class="border-t pt-4 border-slate-600">
          <p class="font-semibold text-slate-300 mb-2">تنظیمات اختصاصی هر پنل:</p>
          <div><label class="font-medium">پروتکل‌ها (برای مرزبان، با | جدا شود):</label><input type="text" name="protocols" placeholder="vless|vmess|trojan" class="w-full p-2 mt-1"></div>
          <div><label class="font-medium">ID اینباند کپی (برای ثنایی):</label><input type="text" name="inbound_copy" class="w-full p-2 mt-1"></div>
          <div><label class="font-medium">لینک نمونه (برای ثنایی):</label><input type="text" name="example_link" class="w-full p-2 mt-1"></div>
        </div>
        <div class="border-t pt-4 border-slate-600">
          <p class="font-semibold text-slate-300 mb-2">موقعیت و ظرفیت</p>
          <div class="grid grid-cols-1 md:grid-cols-3 gap-3">
            <div><label class="font-medium">منطقه:</label><input type="text" name="region" class="w-full p-2 mt-1" placeholder="مثال: آلمان"></div>
            <div><label class="font-medium">حداکثر ظرفیت:</label><input type="number" name="max_capacity" class="w-full p-2 mt-1" value="0" min="0"></div>
            <div><label class="font-medium">مصرف فعلی:</label><input type="number" name="current_load" class="w-full p-2 mt-1" value="0" min="0"></div>
          </div>
          <label class="flex items-center gap-2 mt-3 text-sm"><input type="checkbox" name="maintenance_mode" value="1"><span>فعال‌سازی حالت نگهداری</span></label>
          <textarea name="maintenance_note" class="w-full p-2 mt-2" rows="2" placeholder="یادداشت یا توضیح نگهداری"></textarea>
        </div>
        <div class="border-t pt-4 border-slate-600">
          <p class="font-semibold text-slate-300 mb-2">برچسب‌ها</p>
          <div class="flex flex-wrap gap-2" id="server-tag-options">${buildTagOptions(tags)}</div>
          <p class="text-xs text-slate-400 mt-2">برای مدیریت بهتر از برچسب‌های تعریف‌شده استفاده کنید.</p>
        </div>
        <div class="flex justify-end space-x-3 space-x-reverse pt-6">
          <button type="button" onclick="closeModal()" class="px-5 py-2 rounded-lg transition">انصراف</button>
          <button type="submit" class="bg-indigo-600 hover:bg-indigo-700 text-white px-5 py-2 rounded-lg transition">افزودن</button>
        </div>
      </form>`;
    openModal(html);
    document.getElementById('add-server-form')?.addEventListener('submit', async (e) => {
      e.preventDefault();
      const formData = new FormData(e.target);
      const params = Object.fromEntries(formData.entries());
      params.maintenance_mode = formData.has('maintenance_mode') ? '1' : '0';
      const tagIds = Array.from(e.target.querySelectorAll('input[name="tag_ids[]"]:checked')).map(input => input.value);
      params.tag_ids = tagIds.join(',');
      const r = await apiRequest('add_server', params);
      if (r.success) { closeModal(); loadServers(); } else { alert(r.message || 'خطا در افزودن سرور'); }
    });
  }

  window.openEditServerModal = async function(serverId) {
    const s = window.serverMap?.[serverId];
    if (!s) return;
    const tags = await fetchServerTags();
    const selectedTags = Array.isArray(s.tags) ? s.tags.map(tag => String(tag.id)) : [];
    const html = `
      <h3 class="text-xl font-bold mb-6 border-b pb-3 border-slate-600">ویرایش سرور</h3>
      <form id="edit-server-form" class="space-y-4 text-sm">
        <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
          <div><label class="font-medium">نام سیستمی:</label><input type="text" name="name" class="w-full p-2 mt-1" value="${s.name}" required></div>
          <div><label class="font-medium">نام نمایشی:</label><input type="text" name="display_name" class="w-full p-2 mt-1" value="${s.display_name}" required></div>
        </div>
        <div>
          <label class="font-medium">نوع سرور:</label>
          <select name="type" class="w-full p-2 bg-slate-700 mt-1" required>
            <option value="marzban" ${s.type==='marzban'?'selected':''}>Marzban</option>
            <option value="sanaei" ${s.type==='sanaei'?'selected':''}>Sanaei</option>
          </select>
        </div>
        <div><label class="font-medium">لینک پنل:</label><input type="text" name="link" class="w-full p-2 mt-1" value="${s.link}" required></div>
        <div><label class="font-medium">دامنه جایگزین اشتراک (اختیاری):</label><input type="text" name="custom_domain" class="w-full p-2 mt-1" value="${s.custom_domain || ''}"></div>
        <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
          <div><label class="font-medium">یوزرنیم (اختیاری):</label><input type="text" name="username" class="w-full p-2 mt-1" value="${s.username || ''}"></div>
          <div><label class="font-medium">پسورد (اختیاری):</label><input type="text" name="password" class="w-full p-2 mt-1" value="${s.password || ''}"></div>
        </div>
        <div class="border-t pt-4 border-slate-600">
          <p class="font-semibold text-slate-300 mb-2">تنظیمات اختصاصی هر پنل:</p>
          <div><label class="font-medium">پروتکل‌ها (برای مرزبان، با | جدا شود):</label><input type="text" name="protocols" class="w-full p-2 mt-1" value="${s.protocols || ''}"></div>
          <div><label class="font-medium">ID اینباند کپی (برای ثنایی):</label><input type="text" name="inbound_copy" class="w-full p-2 mt-1" value="${s.inbound_copy || ''}"></div>
          <div><label class="font-medium">لینک نمونه (برای ثنایی):</label><input type="text" name="example_link" class="w-full p-2 mt-1" value="${s.example_link || ''}"></div>
        </div>
        <div class="border-t pt-4 border-slate-600">
          <p class="font-semibold text-slate-300 mb-2">موقعیت و ظرفیت</p>
          <div class="grid grid-cols-1 md:grid-cols-3 gap-3">
            <div><label class="font-medium">منطقه:</label><input type="text" name="region" class="w-full p-2 mt-1" value="${s.region || ''}"></div>
            <div><label class="font-medium">حداکثر ظرفیت:</label><input type="number" name="max_capacity" class="w-full p-2 mt-1" value="${s.max_capacity || 0}" min="0"></div>
            <div><label class="font-medium">مصرف فعلی:</label><input type="number" name="current_load" class="w-full p-2 mt-1" value="${s.current_load || 0}" min="0"></div>
          </div>
          <label class="flex items-center gap-2 mt-3 text-sm"><input type="checkbox" name="maintenance_mode" value="1" ${String(s.maintenance_mode) === '1' ? 'checked' : ''}><span>فعال‌سازی حالت نگهداری</span></label>
          <textarea name="maintenance_note" class="w-full p-2 mt-2" rows="2" placeholder="یادداشت یا توضیح نگهداری">${s.maintenance_note || ''}</textarea>
        </div>
        <div class="border-t pt-4 border-slate-600">
          <p class="font-semibold text-slate-300 mb-2">برچسب‌ها</p>
          <div class="flex flex-wrap gap-2" id="server-edit-tag-options">${buildTagOptions(tags, selectedTags)}</div>
          <p class="text-xs text-slate-400 mt-2">برای حذف تیک را بردارید.</p>
        </div>
        <div class="flex justify-end space-x-3 space-x-reverse pt-6">
          <button type="button" onclick="closeModal()" class="px-5 py-2 rounded-lg transition">انصراف</button>
          <button type="submit" class="bg-indigo-600 hover:bg-indigo-700 text-white px-5 py-2 rounded-lg transition">ذخیره</button>
        </div>
      </form>`;
    openModal(html);
    document.getElementById('edit-server-form')?.addEventListener('submit', async (e) => {
      e.preventDefault();
      const formData = new FormData(e.target);
      const params = Object.fromEntries(formData.entries());
      params.server_id = serverId;
      params.maintenance_mode = formData.has('maintenance_mode') ? '1' : '0';
      const tagIds = Array.from(e.target.querySelectorAll('input[name="tag_ids[]"]:checked')).map(input => input.value);
      params.tag_ids = tagIds.join(',');
      const r = await apiRequest('update_server', params);
      if (r.success) { closeModal(); loadServers(); } else { alert(r.message || 'خطا در بروزرسانی سرور'); }
    });
  }

  // --- Plans page
  async function loadPlans() {
    const list = document.getElementById('plan-list');
    if (!list) return;
    // Use cache if available to avoid flash
    if (cache.plans) {
      list.innerHTML = renderPlansTable(cache.plans);
      wirePlansActions();
      return;
    }
    showLoader(list);
    const data = await apiRequest('get_plans');
    if (!data.success) { list.innerHTML = '<p class="text-center p-4">خطا در بارگذاری تعرفه‌ها.</p>'; return; }
    cache.plans = data.plans || [];
    list.innerHTML = renderPlansTable(cache.plans);
    wirePlansActions();
  }

  // --- Pricing page
  async function loadPricing() {
    const serverSelect = document.getElementById('pricing-server-select');
    const priceList = document.getElementById('price-list');
    if (!serverSelect || !priceList) return;
    // Populate servers fast from cache if available
    if (cache.servers) {
      serverSelect.innerHTML = '<option value="">-- انتخاب کنید --</option>' + cache.servers.filter(s => String(s.is_test_server) !== '1').map(s => `<option value="${s.id}">${s.display_name}</option>`).join('');
    } else {
      const sv = await apiRequest('get_servers');
      cache.servers = (sv.success && sv.servers) ? sv.servers : [];
      serverSelect.innerHTML = cache.servers.length ? ('<option value="">-- انتخاب کنید --</option>' + cache.servers.filter(s => String(s.is_test_server) !== '1').map(s => `<option value="${s.id}">${s.display_name}</option>`).join('')) : '<option>هیچ سروری یافت نشد</option>';
    }

    serverSelect.onchange = async function() {
      const serverId = this.value;
      if (!serverId) { priceList.innerHTML = ''; return; }
      showLoader(priceList);
      const data = await apiRequest('get_prices', { server_id: serverId });
      if (!data.success) { priceList.innerHTML = '<p class="text-center p-4">خطا در دریافت قیمت‌ها.</p>'; return; }
      if (!data.prices || data.prices.length === 0) { priceList.innerHTML = '<p class="text-center p-4">هیچ تعرفه‌ای برای قیمت‌گذاری یافت نشد.</p>'; return; }
      priceList.innerHTML = data.prices.map(item => `
        <div class="flex items-center justify-between border-b border-slate-700 py-4">
          <span class="font-medium">${item.title} (${item.volume}GB, ${item.duration} روز)</span>
          <div class="flex items-center">
            <input type="number" class="w-32 p-2 text-left" id="price-input-${item.plan_id}" value="${item.price || ''}" placeholder="قیمت به تومان">
            <input type="number" class="w-28 p-2 text-left ml-2" id="stars-input-${item.plan_id}" value="${item.stars_price || ''}" placeholder="استارز">
            <button class="bg-indigo-600 text-white px-4 py-2 rounded-lg mr-2 transition hover:bg-indigo-700" data-plan="${item.plan_id}" data-server="${serverId}">ذخیره</button>
          </div>
        </div>`).join('');

      priceList.querySelectorAll('button[data-plan]')?.forEach(btn => btn.addEventListener('click', async () => {
        const planId = btn.getAttribute('data-plan');
        const srvId = btn.getAttribute('data-server');
        const price = document.getElementById(`price-input-${planId}`).value || 0;
        const stars = document.getElementById(`stars-input-${planId}`).value || 0;
        const r = await apiRequest('set_price', { server_id: srvId, plan_id: planId, price, stars_price: stars });
        if (r.success) { btn.classList.add('bg-green-600'); setTimeout(()=>btn.classList.remove('bg-green-600'), 800); }
        else { alert(r.message || 'خطا در ذخیره قیمت'); }
      }));
    };
  }

  // --- Messages page
  async function loadMessages(search = '') {
    const container = document.getElementById('messages-list');
    const searchInput = document.getElementById('messages-search');
    if (!container) return;
    showLoader(container);
    const data = await apiRequest('get_messages', { search });
    if (!data.success) { container.innerHTML = '<p class="text-center p-4">خطا در بارگذاری پیام‌ها.</p>'; return; }
    container.innerHTML = (data.messages || []).map(msg => `
      <div class="border border-slate-700 rounded-lg mb-2 overflow-hidden shadow-sm">
        <div class="bg-white/5 p-4 cursor-pointer flex justify-between items-center transition hover:bg-white/10" onclick="this.nextElementSibling.classList.toggle('hidden'); this.querySelector('i').classList.toggle('rotate-180')">
          <div>
            <h4 class="font-semibold">${msg.message_key}</h4>
            <p class="text-sm text-slate-400">${msg.description || 'بدون توضیحات'}</p>
          </div>
          <i class="fas fa-chevron-down text-slate-400 transition-transform"></i>
        </div>
        <div class="p-4 hidden bg-white/5 border-t border-slate-700">
          <textarea id="msg-val-${msg.message_key}" class="w-full p-2 mb-2" rows="4">${msg.message_value}</textarea>
          <div class="flex justify-end">
            <button class="bg-indigo-600 hover:bg-indigo-700 text-white px-5 py-2 rounded-lg font-semibold transition" data-key="${msg.message_key}">ذخیره</button>
          </div>
        </div>
      </div>`).join('') || '<p class="text-center p-4">هیچ پیامی یافت نشد.</p>';

    container.querySelectorAll('button[data-key]')?.forEach(btn => btn.addEventListener('click', async () => {
      const key = btn.getAttribute('data-key');
      const val = document.getElementById(`msg-val-${key}`).value;
      const r = await apiRequest('update_message', { message_key: key, message_value: val });
      if (r.success) { btn.classList.add('bg-green-600'); setTimeout(()=>btn.classList.remove('bg-green-600'), 800); }
      else { alert('خطا در بروزرسانی پیام.'); }
    }));

    if (searchInput && !searchInput.dataset.wired) {
      searchInput.dataset.wired = '1';
      searchInput.addEventListener('input', () => loadMessages(searchInput.value));
    }
  }

  // --- Settings page (basic fields)
  async function loadSettings() {
    const data = await apiRequest('get_settings');
    if (!data.success) return;
    const s = data.settings || {};
    const setVal = (id, v) => { const el = document.getElementById(id); if (el) el.value = v ?? ''; };
    const setChk = (id, v) => { const el = document.getElementById(id); if (el) el.checked = (v === 'on'); };
    setVal('admin-id-input', s.admin_id);
    setChk('card-payment-status-toggle', s.card_payment_status);
    setVal('card-number-input', s.card_number);
    setChk('crypto-payment-status-toggle', s.crypto_payment_status);
    setVal('crypto-wallets-input', s.crypto_wallets);
    setChk('stars-payment-status-toggle', s.stars_payment_status);
    setChk('kyc-status-toggle', s.kyc_status);
    setVal('kyc-channel-id-input', s.kyc_channel_id);
    setVal('kyc-message-input', s.kyc_message);
    setChk('join-status-toggle', s.join_status);
    setVal('join-channels-input', s.join_channels);
    setChk('bot-status-toggle', s.bot_status);
    setChk('service-image-status-toggle', s.service_image_enabled);
    setChk('payment-image-status-toggle', s.payment_image_enabled);
    setChk('referral-status-toggle', s.referral_status);
    setVal('referral-type-select', s.referral_reward_type);
    setVal('referral-value-input', s.referral_reward_value);
    setVal('referral-min-order-input', s.referral_reward_min_order);
    setVal('referral-expire-input', s.referral_reward_expires_days);
    setVal('referral-prefix-input', s.referral_reward_prefix);
    setVal('referral-limit-input', s.referral_reward_limit);
    document.getElementById('service-image-status-toggle')?.dispatchEvent(new Event('change'));
    document.getElementById('payment-image-status-toggle')?.dispatchEvent(new Event('change'));
    const servPrev = document.getElementById('service-template-preview');
    if (servPrev) {
      if (s.service_template_url) { servPrev.src = s.service_template_url; servPrev.classList.remove('hidden'); }
      else { servPrev.src = ''; servPrev.classList.add('hidden'); }
    }
    const payPrev = document.getElementById('payment-template-preview');
    if (payPrev) {
      if (s.payment_template_url) { payPrev.src = s.payment_template_url; payPrev.classList.remove('hidden'); }
      else { payPrev.src = ''; payPrev.classList.add('hidden'); }
    }
    // Show/hide payment detail sections based on toggles
    const updatePaymentSectionsVisibility = () => {
      const cardOn = document.getElementById('card-payment-status-toggle')?.checked;
      const cryptoOn = document.getElementById('crypto-payment-status-toggle')?.checked;
      const kycOn = document.getElementById('kyc-status-toggle')?.checked;
      const cardDetails = document.getElementById('card-payment-details');
      const cryptoDetails = document.getElementById('crypto-payment-details');
      const kycDetails = document.getElementById('kyc-details');
      if (cardDetails) cardDetails.classList.toggle('hidden', !cardOn);
      if (cryptoDetails) cryptoDetails.classList.toggle('hidden', !cryptoOn);
      if (kycDetails) kycDetails.classList.toggle('hidden', !kycOn);
    };
    ['card-payment-status-toggle','crypto-payment-status-toggle','kyc-status-toggle'].forEach(id => {
      const el = document.getElementById(id);
      if (el && !el.dataset.wired) { el.dataset.wired = '1'; el.addEventListener('change', updatePaymentSectionsVisibility); }
    });
    updatePaymentSectionsVisibility();

    const referralToggle = document.getElementById('referral-status-toggle');
    const referralBody = document.getElementById('referral-settings-body');
    const referralTypeSelect = document.getElementById('referral-type-select');
    const referralValueInput = document.getElementById('referral-value-input');
    const updateReferralVisibility = () => {
      if (referralBody && referralToggle) {
        referralBody.classList.toggle('hidden', !referralToggle.checked);
      }
    };
    if (referralToggle && !referralToggle.dataset.wired) {
      referralToggle.dataset.wired = '1';
      referralToggle.addEventListener('change', updateReferralVisibility);
    }
    const updateReferralValueHint = () => {
      if (!referralTypeSelect || !referralValueInput) return;
      if (referralTypeSelect.value === 'percent') {
        referralValueInput.setAttribute('max', '100');
        referralValueInput.placeholder = 'مثال: 15';
      } else {
        referralValueInput.removeAttribute('max');
        referralValueInput.placeholder = 'مثال: 50000';
      }
    };
    if (referralTypeSelect && !referralTypeSelect.dataset.wired) {
      referralTypeSelect.dataset.wired = '1';
      referralTypeSelect.addEventListener('change', updateReferralValueHint);
    }
    updateReferralVisibility();
    updateReferralValueHint();

    const saveBtn = document.getElementById('save-settings-btn');
    if (saveBtn && !saveBtn.dataset.wired) {
      saveBtn.dataset.wired = '1';
      saveBtn.addEventListener('click', async () => {
        const payload = {
          admin_id: document.getElementById('admin-id-input')?.value || '',
          card_payment_status: document.getElementById('card-payment-status-toggle')?.checked ? 'on' : 'off',
          card_number: document.getElementById('card-number-input')?.value || '',
          crypto_payment_status: document.getElementById('crypto-payment-status-toggle')?.checked ? 'on' : 'off',
          crypto_wallets: document.getElementById('crypto-wallets-input')?.value || '',
          stars_payment_status: document.getElementById('stars-payment-status-toggle')?.checked ? 'on' : 'off',
          kyc_status: document.getElementById('kyc-status-toggle')?.checked ? 'on' : 'off',
          kyc_channel_id: document.getElementById('kyc-channel-id-input')?.value || '',
          kyc_message: document.getElementById('kyc-message-input')?.value || '',
          join_status: document.getElementById('join-status-toggle')?.checked ? 'on' : 'off',
          join_channels: document.getElementById('join-channels-input')?.value || '',
          bot_status: document.getElementById('bot-status-toggle')?.checked ? 'on' : 'off',
          service_image_enabled: document.getElementById('service-image-status-toggle')?.checked ? 'on' : 'off',
          payment_image_enabled: document.getElementById('payment-image-status-toggle')?.checked ? 'on' : 'off',
          referral_status: referralToggle?.checked ? 'on' : 'off',
          referral_reward_type: referralTypeSelect?.value || 'fixed',
          referral_reward_value: referralValueInput?.value || '0',
          referral_reward_min_order: document.getElementById('referral-min-order-input')?.value || '0',
          referral_reward_expires_days: document.getElementById('referral-expire-input')?.value || '0',
          referral_reward_prefix: document.getElementById('referral-prefix-input')?.value || '',
          referral_reward_limit: document.getElementById('referral-limit-input')?.value || '1',
        };
        const r = await apiRequest('update_settings', payload);
        alert(r.success ? 'تنظیمات ذخیره شد.' : (r.message || 'خطا در ذخیره تنظیمات'));
      });
    }
  }

  // --- Image upload helpers ---
  async function compressAndUploadBase64(file) {
    const dataUrl = await new Promise((resolve, reject) => {
      const r = new FileReader();
      r.onerror = () => reject('file read error');
      r.onload = () => resolve(r.result);
      r.readAsDataURL(file);
    });
    const img = await new Promise((resolve, reject) => {
      const im = new Image();
      im.onload = () => resolve(im);
      im.onerror = () => reject('image load error');
      im.src = dataUrl;
    });
    const maxDim = 1200;
    let { width: w, height: h } = img;
    if (w > h && w > maxDim) { h = Math.round(h * (maxDim / w)); w = maxDim; }
    else if (h > w && h > maxDim) { w = Math.round(w * (maxDim / h)); h = maxDim; }
    else if (w === h && w > maxDim) { w = h = maxDim; }
    const canvas = document.createElement('canvas');
    canvas.width = w; canvas.height = h;
    const ctx = canvas.getContext('2d');
    ctx.drawImage(img, 0, 0, w, h);
    let outUrl = canvas.toDataURL('image/webp', 0.92);
    if (outUrl.length > 7.5 * 1024 * 1024) {
      outUrl = canvas.toDataURL('image/jpeg', 0.85);
    }
    const params = new URLSearchParams();
    params.set('action', 'upload_service_template_b64');
    params.set('data', outUrl);
    const resp = await fetch('api.php', {
      method: 'POST',
      body: params,
      credentials: 'same-origin',
      cache: 'no-store',
      headers: { 'Content-Type': 'application/x-www-form-urlencoded' }
    });
    const data = await resp.json();
    if (data.success) {
      const prev = document.getElementById('service-template-preview');
      prev.src = data.url;
      prev.classList.remove('hidden');
      alert('تصویر پس از فشرده‌سازی آپلود شد.');
      return true;
    } else {
      alert(data.message || 'خطا در آپلود (base64)');
      return false;
    }
  }

  async function compressAndUploadPaymentBase64(file) {
    const dataUrl = await new Promise((resolve, reject) => {
      const r = new FileReader();
      r.onerror = () => reject('file read error');
      r.onload = () => resolve(r.result);
      r.readAsDataURL(file);
    });
    const img = await new Promise((resolve, reject) => {
      const im = new Image();
      im.onload = () => resolve(im);
      im.onerror = () => reject('image load error');
      im.src = dataUrl;
    });
    const maxDim = 1200;
    let { width: w, height: h } = img;
    if (w > h && w > maxDim) { h = Math.round(h * (maxDim / w)); w = maxDim; }
    else if (h > w && h > maxDim) { w = Math.round(w * (maxDim / h)); h = maxDim; }
    else if (w === h && w > maxDim) { w = h = maxDim; }
    const canvas = document.createElement('canvas');
    canvas.width = w; canvas.height = h;
    const ctx = canvas.getContext('2d');
    ctx.drawImage(img, 0, 0, w, h);
    let outUrl = canvas.toDataURL('image/webp', 0.92);
    if (outUrl.length > 7.5 * 1024 * 1024) {
      outUrl = canvas.toDataURL('image/jpeg', 0.85);
    }
    const params = new URLSearchParams();
    params.set('action', 'upload_payment_template_b64');
    params.set('data', outUrl);
    const resp = await fetch('api.php', {
      method: 'POST',
      body: params,
      credentials: 'same-origin',
      cache: 'no-store',
      headers: { 'Content-Type': 'application/x-www-form-urlencoded' }
    });
    const data = await resp.json();
    if (data.success) {
      const prev = document.getElementById('payment-template-preview');
      prev.src = data.url;
      prev.classList.remove('hidden');
      alert('تصویر پس از فشرده‌سازی آپلود شد.');
      return true;
    } else {
      alert(data.message || 'خطا در آپلود (base64)');
      return false;
    }
  }

  // --- Settings extra UI for image uploads ---
  function injectServiceTemplateUI() {
    const container = document.getElementById('settings-cards');
    if (!container || document.getElementById('service-template-file')) return;
    let imagesGroup = document.getElementById('images-group');
    if (!imagesGroup) {
      imagesGroup = document.createElement('div');
      imagesGroup.id = 'images-group';
      imagesGroup.className = 'border-t border-slate-600 pt-6';
      imagesGroup.innerHTML = `
        <div class="rounded-xl bg-white/5 border border-white/10 p-4 md:p-5">
          <div class="flex items-center gap-3 mb-4">
            <i class="fas fa-images text-indigo-300"></i>
            <h3 class="font-bold">بخش تصاویر</h3>
          </div>
          <div id="images-grid" class="grid grid-cols-1 lg:grid-cols-2 gap-4"></div>
        </div>`;
      container.appendChild(imagesGroup);
    }
    const imagesGrid = document.getElementById('images-grid') || imagesGroup;
    const block = document.createElement('div');
    block.innerHTML = `
      <div class="rounded-xl bg-white/5 border border-white/10 p-4 md:p-5">
        <div class="flex items-center justify-between mb-4">
          <div class="flex items-center gap-3">
            <i class="fas fa-image text-indigo-300"></i>
            <h3 class="font-bold">تصویر قالب کارت سرویس</h3>
          </div>
          <label class="flex items-center cursor-pointer text-sm">
            <input type="checkbox" id="service-image-status-toggle" class="hidden peer">
            <span class="toggle-track w-12 h-6 rounded-full bg-gray-600 peer-checked:bg-green-500 transition-all duration-300 relative"></span>
            <span class="mr-2">ارسال تصویر</span>
          </label>
        </div>
        <div class="grid grid-cols-1 md:grid-cols-3 gap-4">
          <div class="md:col-span-2">
            <div id="template-drop-zone" class="flex items-center justify-between gap-3 border-2 border-dashed border-slate-500/40 rounded-lg p-4 hover:border-indigo-400/60 transition cursor-pointer bg-white/5">
              <div class="flex items-center gap-3">
                <i class="fas fa-upload text-slate-400"></i>
                <div>
                  <p class="font-medium">تصویر را انتخاب کنید یا بکشید و رها کنید</p>
                  <p class="text-xs text-slate-400">حداکثر ۵ مگابایت • PNG/JPG/WEBP • ترجیحاً مربع (۱:۱)</p>
                </div>
              </div>
              <button type="button" id="choose-template-btn" class="liquid-pill px-3 py-2 text-sm">انتخاب فایل</button>
            </div>
            <input type="file" id="service-template-file" accept="image/png,image/jpeg,image/webp" class="hidden" />
            <div class="mt-3 flex gap-3">
              <button type="button" id="upload-template-btn" class="btn-accent btn-sm">آپلود تصویر</button>
              <button type="button" id="remove-template-btn" class="btn-outline-danger btn-sm">حذف</button>
            </div>
          </div>
          <div class="flex items-center md:justify-end">
            <img id="service-template-preview" src="" alt="Service Template" class="w-36 h-36 object-cover rounded-lg border border-white/10 bg-slate-800/50 hidden" />
          </div>
        </div>
      </div>`;
    (imagesGrid || container).appendChild(block);

    const fileInput = document.getElementById('service-template-file');
    const dropZone = document.getElementById('template-drop-zone');
    const chooseBtn = document.getElementById('choose-template-btn');
    const prev = document.getElementById('service-template-preview');
    const uploadBtn = document.getElementById('upload-template-btn');
    const removeBtn = document.getElementById('remove-template-btn');
    const imgToggle = document.getElementById('service-image-status-toggle');

    function setEnabledControls(enabled) {
      [uploadBtn, removeBtn, chooseBtn, fileInput].forEach(el => { if (el) el.disabled = !enabled; });
      dropZone?.classList.toggle('opacity-50', !enabled);
      dropZone?.classList.toggle('pointer-events-none', !enabled);
    }
    function previewLocalFile(file) {
      if (!file) return;
      const reader = new FileReader();
      reader.onload = () => { prev.src = reader.result; prev.classList.remove('hidden'); };
      reader.readAsDataURL(file);
    }

    chooseBtn?.addEventListener('click', () => fileInput?.click());
    dropZone?.addEventListener('click', () => fileInput?.click());
    dropZone?.addEventListener('dragover', e => { e.preventDefault(); dropZone.classList.add('ring-2','ring-indigo-400/50'); });
    dropZone?.addEventListener('dragleave', () => { dropZone.classList.remove('ring-2','ring-indigo-400/50'); });
    dropZone?.addEventListener('drop', e => {
      e.preventDefault();
      dropZone.classList.remove('ring-2','ring-indigo-400/50');
      const f = e.dataTransfer?.files?.[0];
      if (f) { fileInput.files = e.dataTransfer.files; previewLocalFile(f); }
    });
    fileInput?.addEventListener('change', () => { const f = fileInput.files?.[0]; if (f) previewLocalFile(f); });
    imgToggle?.addEventListener('change', () => setEnabledControls(imgToggle.checked));
    setEnabledControls(imgToggle?.checked ?? true);

    uploadBtn?.addEventListener('click', async () => {
      if (!fileInput.files || !fileInput.files[0]) { alert('لطفاً فایل تصویر را انتخاب کنید.'); return; }
      const fd = new FormData();
      fd.append('action','upload_service_template');
      fd.append('file', fileInput.files[0]);
      try {
        const resp = await fetch('api.php', { method:'POST', body: fd, credentials:'same-origin', cache:'no-store' });
        const data = await resp.json();
        if (data.success) {
          prev.src = data.url;
          prev.classList.remove('hidden');
          alert('تصویر با موفقیت آپلود شد.');
        } else {
          const details = (data.hint || '') + ' ' + (data.message || '');
          if (/upload_max_filesize/i.test(details)) {
            if (confirm('محدودیت upload_max_filesize اجازه آپلود مستقیم نمی‌دهد. تلاش برای فشرده‌سازی و آپلود جایگزین؟')) {
              try { const ok = await compressAndUploadBase64(fileInput.files[0]); if (ok) return; } catch(e) {}
            }
          }
          alert((data.message || 'خطا در آپلود تصویر') + (data.hint ? ('\n' + data.hint) : ''));
        }
      } catch(e) { alert('خطای شبکه/سرور در آپلود.'); }
    });

    removeBtn?.addEventListener('click', async () => {
      if (!confirm('تصویر قالب حذف شود و از پیش‌فرض استفاده گردد؟')) return;
      const fd = new FormData(); fd.append('action','delete_service_template');
      try {
        const resp = await fetch('api.php', { method:'POST', body: fd, credentials:'same-origin', cache:'no-store' });
        const data = await resp.json();
        if (data.success) {
          prev.src = '';
          prev.classList.add('hidden');
          if (fileInput) fileInput.value = '';
          alert('قالب به حالت پیش‌فرض بازگردانده شد.');
        } else {
          alert(data.message || 'خطا در حذف تصویر');
        }
      } catch(e) { alert('خطای شبکه/سرور.'); }
    });
  }

  function injectPaymentTemplateUI() {
    const container = document.getElementById('settings-cards');
    if (!container || document.getElementById('payment-template-file')) return;
    let imagesGroup = document.getElementById('images-group');
    if (!imagesGroup) {
      imagesGroup = document.createElement('div');
      imagesGroup.id = 'images-group';
      imagesGroup.className = 'border-t border-slate-600 pt-6';
      imagesGroup.innerHTML = `
        <div class="rounded-xl bg-white/5 border border-white/10 p-4 md:p-5">
          <div class="flex items-center gap-3 mb-4">
            <i class="fas fa-images text-indigo-300"></i>
            <h3 class="font-bold">بخش تصاویر</h3>
          </div>
          <div id="images-grid" class="grid grid-cols-1 lg:grid-cols-2 gap-4"></div>
        </div>`;
      container.appendChild(imagesGroup);
    }
    const imagesGrid = document.getElementById('images-grid') || imagesGroup;
    const block = document.createElement('div');
    block.innerHTML = `
      <div class="rounded-xl bg-white/5 border border-white/10 p-4 md:p-5">
        <div class="flex items-center justify-between mb-4">
          <div class="flex items-center gap-3">
            <i class="fas fa-image text-pink-300"></i>
            <h3 class="font-bold">تصویر بندانگشتی پرداخت</h3>
          </div>
          <label class="flex items-center cursor-pointer text-sm">
            <input type="checkbox" id="payment-image-status-toggle" class="hidden peer">
            <span class="toggle-track w-12 h-6 rounded-full bg-gray-600 peer-checked:bg-green-500 transition-all duration-300 relative"></span>
            <span class="mr-2">ارسال تصویر</span>
          </label>
        </div>
          <div class="grid grid-cols-1 md:grid-cols-3 gap-4">
            <div class="md:col-span-2">
              <div id="payment-template-drop-zone" class="flex items-center justify-between gap-3 border-2 border-dashed border-slate-500/40 rounded-lg p-4 hover:border-indigo-400/60 transition cursor-pointer bg-white/5">
                <div class="flex items-center gap-3">
                  <i class="fas fa-upload text-slate-400"></i>
                  <div>
                    <p class="font-medium">تصویر را انتخاب کنید یا بکشید و رها کنید</p>
                    <p class="text-xs text-slate-400">حداکثر ۵ مگابایت • PNG/JPG/WEBP • ترجیحاً مربع (۱:۱)</p>
                  </div>
                </div>
                <button type="button" id="choose-payment-template-btn" class="liquid-pill px-3 py-2 text-sm">انتخاب فایل</button>
              </div>
              <input type="file" id="payment-template-file" accept="image/png,image/jpeg,image/webp" class="hidden" />
              <div class="mt-3 flex gap-3">
                <button type="button" id="upload-payment-template-btn" class="btn-accent btn-sm">آپلود تصویر</button>
                <button type="button" id="remove-payment-template-btn" class="btn-outline-danger btn-sm">حذف</button>
              </div>
            </div>
            <div class="flex items-center md:justify-end">
              <img id="payment-template-preview" src="" alt="Payment Template" class="w-36 h-36 object-cover rounded-lg border border-white/10 bg-slate-800/50 hidden" />
            </div>
          </div>
        </div>`;
    (imagesGrid || container).appendChild(block);

    const fileInput = document.getElementById('payment-template-file');
    const dropZone = document.getElementById('payment-template-drop-zone');
    const chooseBtn = document.getElementById('choose-payment-template-btn');
    const prev = document.getElementById('payment-template-preview');
    const uploadBtn = document.getElementById('upload-payment-template-btn');
    const removeBtn = document.getElementById('remove-payment-template-btn');
    const imgToggle = document.getElementById('payment-image-status-toggle');

      function setEnabledControls(enabled) {
        [uploadBtn, removeBtn, chooseBtn, fileInput].forEach(el => { if (el) el.disabled = !enabled; });
        dropZone?.classList.toggle('opacity-50', !enabled);
        dropZone?.classList.toggle('pointer-events-none', !enabled);
      }
      function previewLocalFile(file) {
        if (!file) return;
        const reader = new FileReader();
        reader.onload = e => { prev.src = e.target.result; prev.classList.remove('hidden'); };
        reader.readAsDataURL(file);
      }

    chooseBtn?.addEventListener('click', () => fileInput?.click());
    dropZone?.addEventListener('click', () => fileInput?.click());
    dropZone?.addEventListener('dragover', e => { e.preventDefault(); dropZone.classList.add('ring-2','ring-indigo-400/50'); });
    dropZone?.addEventListener('dragleave', () => { dropZone.classList.remove('ring-2','ring-indigo-400/50'); });
    dropZone?.addEventListener('drop', e => {
      e.preventDefault();
      dropZone.classList.remove('ring-2','ring-indigo-400/50');
      const f = e.dataTransfer?.files?.[0];
      if (f) { fileInput.files = e.dataTransfer.files; previewLocalFile(f); }
    });
    fileInput?.addEventListener('change', () => { const f = fileInput.files?.[0]; if (f) previewLocalFile(f); });
    imgToggle?.addEventListener('change', () => setEnabledControls(imgToggle.checked));
    setEnabledControls(imgToggle?.checked ?? true);

    uploadBtn?.addEventListener('click', async () => {
      if (!fileInput.files || !fileInput.files[0]) { alert('لطفاً فایل تصویر را انتخاب کنید.'); return; }
      const fd = new FormData();
      fd.append('action','upload_payment_template');
      fd.append('file', fileInput.files[0]);
      try {
        const resp = await fetch('api.php', { method:'POST', body: fd, credentials:'same-origin', cache:'no-store' });
        const data = await resp.json();
        if (data.success) {
          prev.src = data.url;
          prev.classList.remove('hidden');
          alert('تصویر با موفقیت آپلود شد.');
        } else {
          const details = (data.hint || '') + ' ' + (data.message || '');
          if (/upload_max_filesize/i.test(details)) {
            if (confirm('محدودیت upload_max_filesize اجازه آپلود مستقیم نمی‌دهد. تلاش برای فشرده‌سازی و آپلود جایگزین؟')) {
              try { const ok = await compressAndUploadPaymentBase64(fileInput.files[0]); if (ok) return; } catch(e) {}
            }
          }
          alert((data.message || 'خطا در آپلود تصویر') + (data.hint ? ('\n' + data.hint) : ''));
        }
      } catch(e) { alert('خطای شبکه/سرور در آپلود.'); }
    });

    removeBtn?.addEventListener('click', async () => {
      if (!confirm('تصویر حذف شود و از پیش‌فرض استفاده گردد؟')) return;
      const fd = new FormData(); fd.append('action','delete_payment_template');
      try {
        const resp = await fetch('api.php', { method:'POST', body: fd, credentials:'same-origin', cache:'no-store' });
        const data = await resp.json();
        if (data.success) {
          prev.src = '';
          prev.classList.add('hidden');
          if (fileInput) fileInput.value = '';
          alert('تصویر حذف شد.');
        } else {
          alert(data.message || 'خطا در حذف تصویر');
        }
      } catch(e) { alert('خطای شبکه/سرور.'); }
    });
  }

  injectServiceTemplateUI();
  injectPaymentTemplateUI();

  // --- Discounts page
  async function loadDiscounts() {
    const list = document.getElementById('discounts-list');
    if (!list) return;
    showLoader(list);
    const data = await apiRequest('get_discounts');
    if (!data.success) { list.innerHTML = '<div class="p-4 text-red-400">خطا در دریافت لیست کدها</div>'; return; }
    list.innerHTML = `
      <table class="min-w-full text-sm table-cool">
        <thead><tr>
          <th class="px-3 py-2 text-right">کد</th>
          <th class="px-3 py-2 text-right">نوع</th>
          <th class="px-3 py-2 text-right">مقدار</th>
          <th class="px-3 py-2 text-right">مصرف</th>
          <th class="px-3 py-2 text-right">سقف کاربر</th>
          <th class="px-3 py-2 text-right">حداقل مبلغ</th>
          <th class="px-3 py-2 text-right">انقضا</th>
          <th class="px-3 py-2 text-right">وضعیت</th>
          <th class="px-3 py-2 text-right">اقدامات</th>
        </tr></thead>
        <tbody>
          ${(data.discounts || []).map(it => `
            <tr>
              <td class="px-3 py-2 font-mono">${it.code}</td>
              <td class="px-3 py-2">${it.type === 'percent' ? 'درصدی' : 'مبلغی'}</td>
              <td class="px-3 py-2">${it.type === 'percent' ? (it.value + '%') : (Number(it.value).toLocaleString('fa-IR') + ' تومان')}</td>
              <td class="px-3 py-2">${(it.used_count ?? 0)} / ${it.max_uses ?? '∞'}</td>
              <td class="px-3 py-2">${it.per_user_limit ?? '-'}</td>
              <td class="px-3 py-2">${it.min_order_amount ? Number(it.min_order_amount).toLocaleString('fa-IR') : '-'}</td>
              <td class="px-3 py-2">${it.expires_at ?? '-'}</td>
              <td class="px-3 py-2">${it.is_active == 1 ? '<span class="text-green-400">فعال</span>' : '<span class="text-slate-400">غیرفعال</span>'}</td>
              <td class="px-3 py-2 flex gap-2">
                <button class="px-3 py-1 text-xs" data-act="toggle" data-id="${it.id}">تغییر وضعیت</button>
                <button class="px-3 py-1 text-xs" data-act="edit" data-id='${it.id}'>ویرایش</button>
                <button class="px-3 py-1 text-xs" data-act="delete" data-id='${it.id}'>حذف</button>
              </td>
            </tr>`).join('')}
        </tbody>
      </table>`;

    // Wire actions
    list.querySelectorAll('button[data-act]')?.forEach(btn => btn.addEventListener('click', async () => {
      const act = btn.getAttribute('data-act');
      const id = btn.getAttribute('data-id');
      if (act === 'toggle') {
        const r = await apiRequest('toggle_discount_status', { id });
        if (r.success) loadDiscounts(); else alert('خطا در تغییر وضعیت');
      } else if (act === 'delete') {
        if (!confirm('حذف این کد؟')) return;
        const r = await apiRequest('delete_discount', { id });
        if (r.success) loadDiscounts(); else alert('خطا در حذف');
      } else if (act === 'edit') {
        openDiscountModal(id, (data.discounts || []).find(x => String(x.id) === String(id)));
      }
    }));

    const addBtn = document.getElementById('add-discount-btn');
    if (addBtn && !addBtn.dataset.wired) {
      addBtn.dataset.wired = '1';
      addBtn.addEventListener('click', () => openDiscountModal(null, null));
    }
  }

  function openDiscountModal(id, values) {
    const v = Object.assign({code:'', type:'percent', value:'', max_uses:'', per_user_limit:'', min_order_amount:'', expires_at:'', is_active:1, note:''}, values || {});
    const html = `
      <h3 class="text-xl font-bold mb-6 border-b pb-3 border-slate-600">${id ? 'ویرایش کد تخفیف' : 'افزودن کد تخفیف'}</h3>
      <form id="discount-form" class="space-y-4">
        <div><label class="font-medium text-sm">کد:</label><input type="text" name="code" class="w-full p-2 mt-1" required value="${v.code}"></div>
        <div class="grid grid-cols-1 md:grid-cols-3 gap-4">
          <div><label class="font-medium text-sm">نوع:</label>
            <select name="type" class="w-full p-2 mt-1">
              <option value="percent" ${v.type==='percent'?'selected':''}>درصدی</option>
              <option value="fixed" ${v.type==='fixed'?'selected':''}>مبلغی</option>
            </select>
          </div>
          <div><label class="font-medium text-sm">مقدار:</label><input type="number" step="0.01" name="value" class="w-full p-2 mt-1" required value="${v.value}"></div>
          <div><label class="font-medium text-sm">حداقل مبلغ سفارش:</label><input type="number" name="min_order_amount" class="w-full p-2 mt-1" value="${v.min_order_amount ?? ''}"></div>
        </div>
        <div class="grid grid-cols-1 md:grid-cols-3 gap-4">
          <div><label class="font-medium text-sm">سقف دفعات (کل):</label><input type="number" name="max_uses" class="w-full p-2 mt-1" value="${v.max_uses ?? ''}"></div>
          <div><label class="font-medium text-sm">سقف برای هر کاربر:</label><input type="number" name="per_user_limit" class="w-full p-2 mt-1" value="${v.per_user_limit ?? ''}"></div>
          <div><label class="font-medium text-sm">انقضا:</label><input type="datetime-local" name="expires_at" class="w-full p-2 mt-1" value="${v.expires_at ? v.expires_at.replace(' ', 'T') : ''}"></div>
        </div>
        <div class="flex items-center gap-2"><input type="checkbox" name="is_active" ${Number(v.is_active)===1?'checked':''}><span>فعال</span></div>
        <div><label class="font-medium text-sm">یادداشت:</label><input type="text" name="note" class="w-full p-2 mt-1" value="${v.note ?? ''}"></div>
        <div class="flex justify-end gap-2 pt-4">
          <button type="button" onclick="closeModal()">انصراف</button>
          <button type="submit">ثبت</button>
        </div>
      </form>`;
    openModal(html);
    document.getElementById('discount-form')?.addEventListener('submit', async (e) => {
      e.preventDefault();
      const params = Object.fromEntries(new FormData(e.target).entries());
      let r;
      if (id) { params.id = id; r = await apiRequest('update_discount', params); }
      else { r = await apiRequest('create_discount', params); }
      if (r.success) { closeModal(); loadDiscounts(); } else { alert(r.message || 'خطا'); }
    });
  }

  function renderPlansTable(plans) {
    if (!plans || plans.length === 0) return '<p class="text-center p-4">هیچ تعرفه‌ای یافت نشد.</p>';
    return `
      <table class="w-full text-right table-cool">
        <thead><tr>
          <th class="p-3">عنوان</th>
          <th class="p-3">حجم (GB)</th>
          <th class="p-3">مدت (روز)</th>
          <th class="p-3">اقدامات</th>
        </tr></thead>
        <tbody>
          ${plans.map(p => `
            <tr>
              <td class="p-4">${p.title}</td>
              <td class="p-4">${p.volume}</td>
              <td class="p-4">${p.duration}</td>
              <td class="p-4"><button class="btn-icon text-red-400 hover:text-red-500" data-del-plan="${p.id}"><i class="fas fa-trash"></i></button></td>
            </tr>`).join('')}
        </tbody>
      </table>`;
  }

  function wirePlansActions() {
    const addBtn = document.getElementById('add-plan-btn');
    if (addBtn && !addBtn.dataset.wired) {
      addBtn.dataset.wired = '1';
      addBtn.addEventListener('click', openAddPlanModal);
    }
    document.querySelectorAll('[data-del-plan]')?.forEach(btn => btn.addEventListener('click', async () => {
      const id = btn.getAttribute('data-del-plan');
      if (!confirm('حذف این تعرفه؟')) return;
      const r = await apiRequest('delete_plan', { plan_id: id });
      if (r.success) { cache.plans = null; loadPlans(); } else { alert(r.message || 'خطا در حذف تعرفه'); }
    }));
  }

  function openAddPlanModal() {
    const html = `
      <h3 class="text-xl font-bold mb-6 border-b pb-3 border-slate-600">افزودن تعرفه جدید</h3>
      <form id="add-plan-form" class="space-y-4">
        <div><label class="font-medium">عنوان تعرفه:</label><input type="text" name="title" class="w-full p-2 mt-1" required></div>
        <div><label class="font-medium">حجم (GB):</label><input type="number" name="volume" class="w-full p-2 mt-1" required></div>
        <div><label class="font-medium">مدت (روز):</label><input type="number" name="duration" class="w-full p-2 mt-1" required></div>
        <div class="flex justify-end space-x-3 space-x-reverse pt-6">
          <button type="button" onclick="closeModal()" class="px-5 py-2 rounded-lg transition">انصراف</button>
          <button type="submit" class="bg-indigo-600 hover:bg-indigo-700 text-white px-5 py-2 rounded-lg transition">افزودن</button>
        </div>
      </form>`;
    openModal(html);
    document.getElementById('add-plan-form')?.addEventListener('submit', async (e) => {
      e.preventDefault();
      const params = Object.fromEntries(new FormData(e.target).entries());
      const r = await apiRequest('add_plan', params);
      if (r.success) { closeModal(); cache.plans = null; loadPlans(); } else { alert(r.message || 'خطا در افزودن تعرفه'); }
    });
  }

  function renderUserDetailsInto(containerId, data) {
    const container = document.getElementById(containerId);
    if (!container) return;
    if (!data.success) { container.innerHTML = `<p class="text-center p-4 text-red-400">${data.message || 'کاربر یافت نشد'}</p>`; return; }
    const user = data.user;
    const services = user.services || [];

    const cards = services.map(s => `
      <div class="bg-white/5 border border-white/10 rounded-xl p-5 shadow-sm transition hover:shadow-md">
        <div class="flex justify-between items-start mb-4">
          <div class="flex-grow">
            <p class="font-bold text-lg">${s.display}</p>
            <p class="text-xs text-slate-400 font-mono">${s.name} | ${s.uuid}</p>
          </div>
          <span class="px-3 py-1 text-xs font-semibold rounded-full ${s.details?.is_active ? 'bg-green-500/20 text-green-300' : 'bg-red-500/20 text-red-300'}">${s.details?.is_active ? 'فعال' : 'غیرفعال'}</span>
        </div>
        <div>
          <div class="flex justify-between items-center mb-1 text-sm">
            <span class="font-medium">مصرف ترافیک</span>
            ${s.details && s.details.used_traffic !== null ? `<span class="font-bold">${s.details.used_traffic} / ${s.size} GB</span>` : `<span class="text-slate-400">نامشخص / ${s.size} GB</span>`}
          </div>
          <div class="w-full bg-slate-700 rounded-full h-2.5">
            <div class="bg-gradient-to-r from-sky-500 to-indigo-500 h-2.5 rounded-full" style="width: ${s.details?.progress_percentage || 0}%"></div>
          </div>
        </div>
        <div class="grid grid-cols-2 gap-4 text-sm mt-5 pt-4 border-t border-white/10">
          <div class="flex items-center"><i class="fas fa-server ml-2 text-slate-400"></i><strong>سرور:</strong><span class="mr-2">${s.location}</span></div>
          <div class="flex items-center"><i class="fas fa-calendar-plus ml-2 text-slate-400"></i><strong>تاریخ ساخت:</strong><span class="mr-2">${s.details?.created_at_jalali || '-'}</span></div>
          <div class="flex items-center"><i class="fas fa-hourglass-half ml-2 text-slate-400"></i><strong>روز مانده:</strong><span class="mr-2">${s.details?.days_remaining ?? 0} روز</span></div>
          <div class="flex items-center"><i class="fas fa-calendar-times ml-2 text-slate-400"></i><strong>تاریخ انقضا:</strong><span class="mr-2">${s.details?.expire_date_jalali || '-'}</span></div>
        </div>
        <div class="text-left mt-4">
          <button class="text-red-400 hover:text-red-500 text-sm" onclick="deleteService(${s.id}, false)"><i class="fas fa-trash-alt ml-1"></i>حذف سرویس</button>
        </div>
      </div>
    `).join('');

    container.innerHTML = `
      <div class="space-y-6">
        <div class="bg-white/10 p-6 rounded-xl border border-white/10 shadow-sm">
          <div class="grid grid-cols-1 md:grid-cols-2 gap-x-6 gap-y-4">
            <div class="flex items-center"><i class="fas fa-id-card text-lg ml-3 text-slate-400"></i><p><strong>ID کاربر:</strong> <code class="bg-slate-700 p-1 rounded-md text-sm">${user.id}</code></p></div>
            <div class="flex items-center"><i class="fab fa-telegram-plane text-lg ml-3 text-slate-400"></i><p><strong>نام کاربری:</strong> ${user.username ? '@' + user.username : 'ندارد'}</p></div>
          </div>
          <div class="border-t border-white/10 pt-4 mt-4">
            <p class="font-semibold text-lg flex items-center"><i class="fas fa-wallet text-lg ml-3 text-slate-400"></i><strong>موجودی:</strong><span class="mr-2 text-xl font-bold text-green-400">${new Intl.NumberFormat('fa-IR').format(user.coin)}</span><span class="text-sm mr-1">تومان</span></p>
            <div class="flex items-center space-x-2 space-x-reverse mt-3">
              <input type="number" id="balance-amount-${user.id}" class="p-2 w-40" placeholder="مبلغ به تومان">
              <button class="bg-green-600 text-white px-3 py-2 rounded-lg text-sm transition hover:bg-green-700" onclick="updateBalance('${user.id}', 'add')">افزایش</button>
              <button class="bg-red-600 text-white px-3 py-2 rounded-lg text-sm transition hover:bg-red-700" onclick="updateBalance('${user.id}', 'subtract')">کاهش</button>
            </div>
          </div>
        </div>
        <div>
          <h4 class="font-bold text-2xl mb-4">سرویس‌های کاربر (${services.length})</h4>
          <div class="space-y-5">${cards || '<p class="text-center p-3">سرویسی ندارد.</p>'}</div>
        </div>
      </div>`;
  }
  // --- Users page filters wiring
  const usersSearch = document.getElementById('users-list-search');
  const usersStatus = document.getElementById('users-list-status-filter');
  const usersServer = document.getElementById('users-list-server-filter');
  function applyUsersFilters() {
    loadUsersList(1, usersSearch?.value || '', usersStatus?.value || '', usersServer?.value || '');
  }
  usersSearch?.addEventListener('input', applyUsersFilters);
  usersStatus?.addEventListener('change', applyUsersFilters);
  usersServer?.addEventListener('change', applyUsersFilters);

  // Populate server filter selects
  (async () => {
    try {
      const d = await apiRequest('get_servers');
      if (d.success && d.servers) {
        const opts = ['<option value="">همه سرورها</option>'].concat(d.servers.map(s => `<option value="${s.name}">${s.display_name}</option>`)).join('');
        if (usersServer) usersServer.innerHTML = opts;
        const servicesServer = document.getElementById('services-list-server-filter');
        if (servicesServer) servicesServer.innerHTML = opts;
      }
    } catch {}
  })();

  // Services filters
  (function wireServicesFilters(){
    const searchInput = document.getElementById('services-list-search');
    const serverFilter = document.getElementById('services-list-server-filter');
    const statusFilter = document.getElementById('services-list-status-filter');
    const apply = () => loadServicesList(1, searchInput?.value || '', serverFilter?.value || '', statusFilter?.value || '');
    searchInput?.addEventListener('input', apply);
    serverFilter?.addEventListener('change', apply);
    statusFilter?.addEventListener('change', apply);
  })();

  // Expose minimal actions referenced in HTML if needed
  window.updateBalance = async (userId, type) => {
    const amount = document.getElementById(`balance-amount-${userId}`)?.value;
    if (!amount || amount <= 0) { alert('لطفا مبلغ معتبر وارد کنید'); return; }
    const r = await apiRequest('update_balance', { user_id: userId, amount, type });
    alert(r.success ? 'انجام شد' : (r.message || 'خطا'));
  };

  window.deleteService = async (serviceId, refreshList) => {
    if (!confirm('حذف سرویس انجام شود؟')) return;
    const r = await apiRequest('delete_service', { service_id: serviceId });
    alert(r.success ? 'حذف شد' : (r.message || 'خطا'));
    if (refreshList) {
      const ss = document.getElementById('services-list-search')?.value || '';
      const sf = document.getElementById('services-list-server-filter')?.value || '';
      const st = document.getElementById('services-list-status-filter')?.value || '';
      loadServicesList(1, ss, sf, st);
    }
  };

  // --- Navigation
  function showPage(id) {
    pages.forEach(p => p.classList.toggle('hidden', p.id !== id));
  }
  function loadPageContent(id) {
    const map = {
      'dashboard': loadDashboard,
      'users-list': () => loadUsersList(1, usersSearch?.value || '', usersStatus?.value || '', usersServer?.value || ''),
      'services-list': () => loadServicesList(),
      'user-search': () => wireUserSearch(),
      'servers': () => loadServers(),
      'plans': () => loadPlans(),
      'pricing': () => loadPricing(),
      'discounts': () => loadDiscounts(),
      'messages': () => loadMessages(),
      'broadcast': () => wireBroadcast(),
      'settings': () => loadSettings()
    };
    map[id]?.();
  }

  sidebarLinks.forEach(a => a.addEventListener('click', (e) => {
    e.preventDefault();
    sidebarLinks.forEach(l => l.classList.remove('active'));
    a.classList.add('active');
    const id = a.getAttribute('data-target');
    showPage(id);
    loadPageContent(id);
  }));

  // Initial
  const initial = document.querySelector('.sidebar-link.active')?.getAttribute('data-target') || 'dashboard';
  showPage(initial);
  // Preload critical data to reduce perceived latency
  (async () => {
    try {
      const [sv, pl] = await Promise.all([
        apiRequest('get_servers'),
        apiRequest('get_plans')
      ]);
      if (sv.success) cache.servers = sv.servers || [];
      if (pl.success) cache.plans = pl.plans || [];
    } catch {}
    loadPageContent(initial);
  })();

  // --- User search page wiring
  function wireUserSearch() {
    const input = document.getElementById('user-search-input');
    const container = document.getElementById('user-details-container');
    if (!input || !container) return;
    container.innerHTML = '<p class="text-center py-8">برای مشاهده اطلاعات، کاربر را جستجو کنید.</p>';
    if (input.dataset.wired) return; input.dataset.wired = '1';
    let t; input.addEventListener('input', async function(){
      const q = this.value.trim();
      if (q.length < 3) { container.innerHTML = '<p class="text-center py-8">برای جستجو حداقل ۳ کاراکتر وارد کنید.</p>'; return; }
      clearTimeout(t); t = setTimeout(async () => {
        showLoader(container);
        const d = await apiRequest('get_user_details', { identifier: q });
        renderUserDetailsInto('user-details-container', d);
      }, 300);
    });
  }

  // --- Broadcast wiring
  function wireBroadcast() {
    const btn = document.getElementById('send-broadcast-btn');
    if (!btn || btn.dataset.wired) return;
    btn.dataset.wired = '1';
    btn.addEventListener('click', async () => {
      const message = document.getElementById('broadcast-message')?.value || '';
      if (!message.trim()) { alert('لطفاً متن پیام را وارد کنید.'); return; }
      if (!confirm('ارسال پیام به همه کاربران انجام شود؟')) return;
      const status = document.getElementById('broadcast-status');
      if (status) status.innerHTML = '<p class="text-blue-400 font-medium">در حال ارسال...</p>';
      const r = await apiRequest('broadcast', { message });
      if (status) status.innerHTML = r.success ? '<p class="text-green-400 font-medium">پیام در صف ارسال قرار گرفت.</p>' : `<p class="text-red-400 font-medium">${r.message || 'خطا در ارسال'}</p>`;
    });
  }

  // Help variables modal for messages page
  const helpBtn = document.getElementById('help-variables-btn');
  helpBtn?.addEventListener('click', () => {
    openModal(`
      <div class="flex justify-between items-center border-b border-slate-600 pb-3 mb-4">
        <h3 class="text-xl font-bold">راهنمای متغیرهای پیام</h3>
        <button onclick="closeModal()" class="text-2xl">&times;</button>
      </div>
      <div class="max-h-[70vh] overflow-y-auto pr-2 space-y-3 text-sm">
        <p><code>{service_name}</code> - نام نمایشی سرویس</p>
        <p><code>{new_name}</code> - نام جدید سرویس</p>
        <p><code>{location}</code> - نام لوکیشن/سرور</p>
        <p><code>{size}</code> - حجم کل سرویس</p>
        <p><code>{amount}</code> - قیمت سرویس</p>
        <p><code>{days}</code> - مدت سرویس (روز)</p>
        <p><code>{date}</code> - تاریخ انقضا (شمسی)</p>
        <p><code>{status}</code> - وضعیت سرویس</p>
        <p><code>{used}</code> - حجم مصرف‌شده</p>
        <p><code>{total}</code> - حجم کل</p>
        <p><code>{percent}</code> - درصد مصرف</p>
        <p><code>{progress_bar}</code> - نوار گرافیکی مصرف</p>
        <p><code>{user_id}</code> - آیدی عددی کاربر</p>
        <p><code>{service_count}</code> - تعداد سرویس‌های کاربر</p>
        <p><code>{order_code}</code> - کد سفارش</p>
        <p><code>{card_info}</code> - اطلاعات کارت</p>
      </div>`);
  });
});
