<?php
// Lightweight image helper to render service details over a template image.
// Works with Imagick if available, otherwise falls back to GD.

function ih_find_font() {
    $candidates = [
        __DIR__ . '/../assets/fonts/Vazirmatn-Regular.ttf',
        __DIR__ . '/../assets/fonts/Vazirmatn-Medium.ttf',
        __DIR__ . '/../assets/fonts/NotoNaskhArabic-Regular.ttf',
        __DIR__ . '/../assets/fonts/NotoSansArabic-Regular.ttf',
        __DIR__ . '/../assets/fonts/IRANSans.ttf',
        '/usr/share/fonts/truetype/dejavu/DejaVuSans.ttf',
        '/usr/share/fonts/truetype/dejavu/DejaVuSansCondensed.ttf',
        '/usr/share/fonts/truetype/noto/NotoNaskhArabic-Regular.ttf',
        '/usr/share/fonts/truetype/noto/NotoSansArabic-Regular.ttf',
        '/usr/share/fonts/truetype/freefont/FreeSans.ttf',
        'C:\\Windows\\Fonts\\Tahoma.ttf',
        'C:\\Windows\\Fonts\\arial.ttf',
    ];
    foreach ($candidates as $f) { if (is_readable($f)) return $f; }
    return null;
}

function ih_find_pango_family() {
    // Prefer widely available Arabic/Persian families by name (must be installed on host)
    $families = [
        'Vazirmatn',
        'Noto Naskh Arabic',
        'Noto Sans Arabic',
        'IRANSans',
        'Tahoma',
        'Arial',
        'Sans'
    ];
    // We cannot enumerate installed fonts reliably from PHP; return first candidate
    return $families[0];
}

function ih_rtl_wrap($s) {
    // Force RTL embedding for consistent bidi rendering
    $RLE = "\xE2\x80\xAB"; // U+202B
    $PDF = "\xE2\x80\xAC"; // U+202C
    return $RLE . $s . $PDF;
}

function ih_contains_arabic($s) {
    // Detect if a string contains any Arabic-script characters
    return (bool)preg_match('/\p{Arabic}/u', $s);
}

function ih_has_pango() {
    if (!extension_loaded('imagick')) return false;
    try {
        $t = new Imagick();
        $t->readImage("pango:تست");
        $t->destroy();
        return true;
    } catch (Throwable $e) {
        return false;
    }
}

// --- Arabic shaping fallback (when Pango is not available) ---
// Basic joining logic for Persian/Arabic letters using Presentation Forms.
// Note: This is a pragmatic fallback and does not implement full Unicode bidi.

function ih_utf8_chars($str) {
    preg_match_all('/./us', $str, $m);
    return $m[0];
}

function ih_utf8_strrev($str) {
    $chars = ih_utf8_chars($str);
    return implode('', array_reverse($chars));
}

function ih_is_arabic_char($ch) {
    return (bool)preg_match('/\p{Arabic}/u', $ch);
}

function ih_arabic_shaper_map() {
    // Map base -> [isolated, final, initial, medial]
    static $map = null;
    if ($map) return $map;
    $m = [];
    $m['ا'] = ["\u{FE8D}", "\u{FE8E}"]; // no initial/medial
    $m['آ'] = ["\u{FE81}", "\u{FE82}"];
    $m['أ'] = ["\u{FE83}", "\u{FE84}"];
    $m['إ'] = ["\u{FE87}", "\u{FE88}"];
    $m['ء'] = ["\u{FE80}"];
    $m['ؤ'] = ["\u{FE85}", "\u{FE86}"];
    $m['ئ'] = ["\u{FE89}", "\u{FE8A}", "\u{FE8B}", "\u{FE8C}"];
    $m['ب'] = ["\u{FE8F}", "\u{FE90}", "\u{FE91}", "\u{FE92}"];
    $m['پ'] = ["\u{FB56}", "\u{FB57}", "\u{FB58}", "\u{FB59}"];
    $m['ت'] = ["\u{FE95}", "\u{FE96}", "\u{FE97}", "\u{FE98}"];
    $m['ث'] = ["\u{FE99}", "\u{FE9A}", "\u{FE9B}", "\u{FE9C}"];
    $m['ج'] = ["\u{FE9D}", "\u{FE9E}", "\u{FE9F}", "\u{FEA0}"];
    $m['چ'] = ["\u{FB7A}", "\u{FB7B}", "\u{FB7C}", "\u{FB7D}"];
    $m['ح'] = ["\u{FEA1}", "\u{FEA2}", "\u{FEA3}", "\u{FEA4}"];
    $m['خ'] = ["\u{FEA5}", "\u{FEA6}", "\u{FEA7}", "\u{FEA8}"];
    $m['د'] = ["\u{FEA9}", "\u{FEAA}"];
    $m['ذ'] = ["\u{FEAB}", "\u{FEAC}"];
    $m['ر'] = ["\u{FEAD}", "\u{FEAE}"];
    $m['ز'] = ["\u{FEAF}", "\u{FEB0}"];
    $m['ژ'] = ["\u{FB8A}", "\u{FB8B}"];
    $m['س'] = ["\u{FEB1}", "\u{FEB2}", "\u{FEB3}", "\u{FEB4}"];
    $m['ش'] = ["\u{FEB5}", "\u{FEB6}", "\u{FEB7}", "\u{FEB8}"];
    $m['ص'] = ["\u{FEB9}", "\u{FEBA}", "\u{FEBB}", "\u{FEBC}"];
    $m['ض'] = ["\u{FEBD}", "\u{FEBE}", "\u{FEBF}", "\u{FEC0}"];
    $m['ط'] = ["\u{FEC1}", "\u{FEC2}", "\u{FEC3}", "\u{FEC4}"];
    $m['ظ'] = ["\u{FEC5}", "\u{FEC6}", "\u{FEC7}", "\u{FEC8}"];
    $m['ع'] = ["\u{FEC9}", "\u{FECA}", "\u{FECB}", "\u{FECC}"];
    $m['غ'] = ["\u{FECD}", "\u{FECE}", "\u{FECF}", "\u{FED0}"];
    $m['ف'] = ["\u{FED1}", "\u{FED2}", "\u{FED3}", "\u{FED4}"];
    $m['ق'] = ["\u{FED5}", "\u{FED6}", "\u{FED7}", "\u{FED8}"];
    $m['ک'] = ["\u{FB8E}", "\u{FB8F}", "\u{FB90}", "\u{FB91}"]; // Persian Kaf
    $m['گ'] = ["\u{FB92}", "\u{FB93}", "\u{FB94}", "\u{FB95}"];
    $m['ل'] = ["\u{FEDD}", "\u{FEDE}", "\u{FEDF}", "\u{FEE0}"];
    $m['م'] = ["\u{FEE1}", "\u{FEE2}", "\u{FEE3}", "\u{FEE4}"];
    $m['ن'] = ["\u{FEE5}", "\u{FEE6}", "\u{FEE7}", "\u{FEE8}"];
    $m['ه'] = ["\u{FEE9}", "\u{FEEA}", "\u{FEEB}", "\u{FEEC}"];
    $m['و'] = ["\u{FEED}", "\u{FEEE}"];
    $m['ی'] = ["\u{FBFC}", "\u{FBFD}", "\u{FBFE}", "\u{FBFF}"]; // Persian Yeh
    $m['ى'] = ["\u{FEEF}", "\u{FEF0}", "\u{FEF3}", "\u{FEF4}"]; // Arabic Alef Maqsura forms as fallback
    $m['ة'] = ["\u{FE93}", "\u{FE94}"];
    return $map = $m;
}

function ih_non_connecting_left() {
    // Letters that do not connect to the next letter (left-joining)
    return ['ا','آ','أ','إ','ء','ؤ','ئ','د','ذ','ر','ز','ژ','و'];
}

function ih_can_join_prev($ch) { return ih_is_arabic_char($ch); }
function ih_can_join_next($ch) { return ih_is_arabic_char($ch) && !in_array($ch, ih_non_connecting_left(), true); }

function ih_shape_arabic_word($word) {
    $map = ih_arabic_shaper_map();
    $chars = ih_utf8_chars($word);
    // Handle lam-alef ligature simple replacement
    for ($i = 0; $i < count($chars) - 1; $i++) {
        if ($chars[$i] === 'ل' && in_array($chars[$i+1], ['ا','آ','أ','إ'], true)) {
            $chars[$i] = 'ﻻ'; // basic lam-alef ligature
            array_splice($chars, $i+1, 1);
        }
    }
    $out = '';
    $n = count($chars);
    for ($i = 0; $i < $n; $i++) {
        $ch = $chars[$i];
        if (!isset($map[$ch])) { $out .= $ch; continue; }
        $prev = $i > 0 ? $chars[$i-1] : '';
        $next = $i < $n-1 ? $chars[$i+1] : '';
        $joinPrev = ih_can_join_prev($prev) && ih_can_join_next($prev);
        $joinNext = ih_can_join_next($ch) && ih_can_join_prev($next);
        $forms = $map[$ch];
        // Select form index based on joining
        // forms: [isolated, final, initial, medial]
        if ($joinPrev && $joinNext && isset($forms[3])) $glyph = $forms[3];
        elseif ($joinPrev && isset($forms[1])) $glyph = $forms[1];
        elseif ($joinNext && isset($forms[2])) $glyph = $forms[2];
        else $glyph = $forms[0];
        $out .= json_decode('"' . $glyph . '"');
    }
    return $out;
}

function ih_bidi_rtl_line($text) {
    // Split into Arabic vs non-Arabic segments (keep original order!),
    // shape Arabic segments and reverse within segment for LTR drawing.
    preg_match_all('/(\p{Arabic}+|[^\p{Arabic}]+)/u', $text, $m);
    $segs = $m[0] ?? [$text];
    $out = '';
    foreach ($segs as $seg) {
        if (preg_match('/^\p{Arabic}+$/u', $seg)) {
            $shaped = ih_shape_arabic_word($seg);
            $out .= ih_utf8_strrev($shaped);
        } else {
            $out .= $seg; // keep Latin/nums/punctuation as-is
        }
    }
    return $out;
}

function ih_default_template($w = 1024, $h = 1024) {
    // Create a simple gradient bg with rounded rectangle in center
    if (!extension_loaded('gd')) return null;
    $im = imagecreatetruecolor($w, $h);
    imageantialias($im, true);
    // Gradient background
    for ($y = 0; $y < $h; $y++) {
        $ratio = $y / max(1, $h - 1);
        $r = 10 + (int)(30 * $ratio);
        $g = 60 + (int)(70 * $ratio);
        $b = 120 + (int)(100 * $ratio);
        $col = imagecolorallocate($im, $r, $g, $b);
        imageline($im, 0, $y, $w, $y, $col);
    }
    // Rounded rectangle parameters
    $pad = (int)($w * 0.12);
    $rx = $pad; $ry = (int)($h * 0.13);
    $rw = $w - 2 * $pad; $rh = $h - 2 * $ry;
    $radius = (int)min($rw, $rh) * 0.06;
    $rect = imagecreatetruecolor($w, $h);
    imagesavealpha($rect, true);
    $transparent = imagecolorallocatealpha($rect, 0, 0, 0, 127);
    imagefill($rect, 0, 0, $transparent);
    $white = imagecolorallocatealpha($rect, 255, 255, 255, 90);
    // Draw rounded rectangle on overlay
    // Utility to draw rounded rectangle
    imagefilledrectangle($rect, $rx + $radius, $ry, $rx + $rw - $radius, $ry + $rh, $white);
    imagefilledrectangle($rect, $rx, $ry + $radius, $rx + $rw, $ry + $rh - $radius, $white);
    imagefilledellipse($rect, $rx + $radius, $ry + $radius, $radius * 2, $radius * 2, $white);
    imagefilledellipse($rect, $rx + $rw - $radius, $ry + $radius, $radius * 2, $radius * 2, $white);
    imagefilledellipse($rect, $rx + $radius, $ry + $rh - $radius, $radius * 2, $radius * 2, $white);
    imagefilledellipse($rect, $rx + $rw - $radius, $ry + $rh - $radius, $radius * 2, $radius * 2, $white);
    // Merge overlay
    imagecopy($im, $rect, 0, 0, 0, 0, $w, $h);
    imagedestroy($rect);
    // Save temp file
    $tmp = sys_get_temp_dir() . '/svc_template_' . uniqid() . '.png';
    imagepng($im, $tmp, 6);
    imagedestroy($im);
    return $tmp;
}

function ih_wrap_text($font, $fontSize, $text, $maxWidth) {
    // Improved wrapping: keep "label: value" on the same line
    // to avoid breaking after a colon (e.g., "Service: a123...")
    if (strpos($text, ':') !== false) {
        return [trim($text)];
    }
    // naive word wrap based on bounding box for other cases
    $words = preg_split('/\s+/u', trim($text));
    $lines = [];
    $current = '';
    foreach ($words as $w) {
        $try = trim($current === '' ? $w : ($current . ' ' . $w));
        $box = imagettfbbox($fontSize, 0, $font, $try);
        $wpx = abs($box[4] - $box[0]);
        if ($wpx <= $maxWidth) {
            $current = $try;
        } else {
            if ($current !== '') $lines[] = $current;
            $current = $w;
        }
    }
    if ($current !== '') $lines[] = $current;
    return $lines;
}

// Compute GD text metrics for a set of lines
function ih_gd_metrics($fontPath, $fontSize, array $lines, $lineSpacing) {
    $maxW = 0;
    foreach ($lines as $ln) {
        $bbox = imagettfbbox($fontSize, 0, $fontPath, $ln);
        $wpx = abs($bbox[4] - $bbox[0]);
        if ($wpx > $maxW) $maxW = $wpx;
    }
    $lineHeight = $fontSize * $lineSpacing;
    $totalH = count($lines) * $lineHeight;
    return [$maxW, $totalH, $lineHeight];
}

// Find a font size that fits both width and height constraints (GD path)
function ih_gd_fit_font($fontPath, array $lines, $boxW, $boxH, $initialFontSize, $lineSpacing) {
    $fs = (int)$initialFontSize;
    if ($fs <= 0) $fs = 16;
    // Iterate a few times to converge
    for ($i = 0; $i < 5; $i++) {
        list($maxW, $totalH) = ih_gd_metrics($fontPath, $fs, $lines, $lineSpacing);
        $needScaleW = $maxW > 0 ? ($boxW / $maxW) : 1.0;
        $needScaleH = $totalH > 0 ? ($boxH / $totalH) : 1.0;
        $scale = min($needScaleW, $needScaleH, 1.0);
        $newFs = (int)floor($fs * $scale);
        if ($newFs < 12) $newFs = 12; // minimum for readability
        if ($newFs == $fs) break;
        $fs = $newFs;
    }
    return $fs;
}

function generate_service_card_image($template_path, $output_path, array $lines, $options = []) {
    // Try Imagick first
    $font = $options['font'] ?? ih_find_font();
    $color = $options['color'] ?? '#0b2a43';
    $fontSize = $options['font_size'] ?? 42;
    $lineSpacing = $options['line_spacing'] ?? 1.35; // multiplier

    if (extension_loaded('imagick')) {
        try {
            $img = new Imagick();
            if ($template_path && is_readable($template_path)) $img->readImage($template_path);
            else {
                $fallback = ih_default_template();
                $img->readImage($fallback);
            }
            $w = $img->getImageWidth();
            $h = $img->getImageHeight();
            // Text area inside rounded rectangle
            $left = (int)($w * 0.20);
            $right = (int)($w * 0.80);
            $top = (int)($h * 0.22);
            $bottom = (int)($h * 0.78);
            $textTop = $top; // avatar disabled by request
            // Try Pango if available for accurate RTL shaping
            if (ih_has_pango()) {
                $boxW = max(200, $right - $left);
                $boxH = max(200, $bottom - $textTop);
                $fontFamily = ih_find_pango_family();
                $size = (int)($fontSize ?: max(40, (int)($w*0.055)));
                $markupLines = [];
                $hasArabic = false;
                foreach ($lines as $ln) {
                    if (!$hasArabic && ih_contains_arabic($ln)) { $hasArabic = true; }
                    $markupLines[] = htmlspecialchars($ln, ENT_QUOTES | ENT_SUBSTITUTE, 'UTF-8');
                }
                $sep = ($lineSpacing ?? 1.35) >= 1.5 ? "\n\n" : "\n";
                $markup = implode($sep, $markupLines);
                $pango = new Imagick();
                $pango->setBackgroundColor(new ImagickPixel('transparent'));
                // pango supports markup like <span ...>; wrap only if Arabic exists
                $content = $hasArabic ? ih_rtl_wrap($markup) : $markup;
                $pango->readImage("pango:<span font='".$fontFamily." " . $size . "' foreground='" . ltrim($color,'#') . "'>".$content."</span>");
                $pw = max(1, $pango->getImageWidth());
                $ph = max(1, $pango->getImageHeight());
                $scale = min(1.0, $boxW / $pw, $boxH / $ph);
                if ($scale < 1.0) {
                    $newW = (int)floor($pw * $scale);
                    $newH = (int)floor($ph * $scale);
                    if ($newW <= 0) $newW = 1; if ($newH <= 0) $newH = 1;
                    $pango->resizeImage($newW, $newH, Imagick::FILTER_LANCZOS, 1);
                }
                $x = (int)(($left + $right - $pango->getImageWidth()) / 2);
                $y = (int)(($textTop + $bottom - $pango->getImageHeight()) / 2);
                $img->compositeImage($pango, Imagick::COMPOSITE_OVER, $x, $y);
                $pango->destroy();
            } else {
                // fallback to manual annotate (no shaping but will render if font supports it)
                $draw = new ImagickDraw();
                $draw->setFillColor(new ImagickPixel($color));
                if ($font && is_readable($font)) $draw->setFont($font);
                $draw->setTextAlignment(Imagick::ALIGN_CENTER);
                if (!isset($options['font_size'])) { $fontSize = max(28, (int)($w * 0.050)); }
                // Prepare shaped lines once
                $shapedLines = [];
                foreach ($lines as $ln) { $shapedLines[] = ih_bidi_rtl_line($ln); }
                // Fit font size to box
                $boxW = max(50, $right - $left);
                $boxH = max(50, $bottom - $top);
                for ($i=0; $i<5; $i++) {
                    $draw->setFontSize($fontSize);
                    $maxW = 0;
                    foreach ($shapedLines as $ln) {
                        $m = $img->queryFontMetrics($draw, $ln);
                        $maxW = max($maxW, (int)ceil($m['textWidth']));
                    }
                    $lineHeight = $fontSize * $lineSpacing;
                    $totalHeight = count($shapedLines) * $lineHeight;
                    if ($maxW <= $boxW && $totalHeight <= $boxH) break;
                    $scaleW = $boxW / max(1, $maxW);
                    $scaleH = $boxH / max(1, $totalHeight);
                    $scale = min($scaleW, $scaleH, 1.0);
                    $newSize = (int)floor($fontSize * $scale);
                    if ($newSize < 12) { $fontSize = 12; break; }
                    if ($newSize == $fontSize) { $fontSize--; if ($fontSize < 12) { $fontSize = 12; break; } }
                    else { $fontSize = $newSize; }
                }
                $draw->setFontSize($fontSize);
                $lineHeight = $fontSize * $lineSpacing;
                $totalHeight = count($shapedLines) * $lineHeight;
                $y = (int)(($textTop + $bottom) / 2 - $totalHeight / 2 + $fontSize);
                foreach ($shapedLines as $ln) {
                    $draw->annotation((int)(($left + $right) / 2), (int)$y, $ln);
                    $y += $lineHeight;
                }
                $img->drawImage($draw);
            }
            $img->setImageFormat('jpeg');
            $img->setImageCompressionQuality(90);
            $img->writeImage($output_path);
            $img->destroy();
            return true;
        } catch (Throwable $e) {
            // fall through to GD
        }
    }

    if (!extension_loaded('gd')) return false;
    // GD fallback
    $src = null;
    if ($template_path && is_readable($template_path)) {
        $ext = strtolower(pathinfo($template_path, PATHINFO_EXTENSION));
        if ($ext === 'png') $src = imagecreatefrompng($template_path);
        elseif ($ext === 'gif') $src = imagecreatefromgif($template_path);
        else $src = imagecreatefromjpeg($template_path);
    }
    if (!$src) {
        $tpl = ih_default_template();
        $src = imagecreatefrompng($tpl);
    }
    $w = imagesx($src); $h = imagesy($src);
    $left = (int)($w * 0.20);
    $right = (int)($w * 0.80);
    $top = (int)($h * 0.22);
    $bottom = (int)($h * 0.78);
    $textTop = $top; // avatar disabled
    if (!isset($options['font_size'])) {
        $fontSize = max(26, (int)($w * 0.050));
    }
    $maxWidth = $right - $left;
    $textColor = imagecolorallocate($src, 20, 50, 75); // darker for readability
    $fontPath = $font;
    // Avatar rendering disabled
    $allLines = [];
    if ($fontPath && is_readable($fontPath)) {
        // Build shaped base lines (no wrap for label: value)
        $baseLines = [];
        foreach ($lines as $ln) { $baseLines[] = ih_bidi_rtl_line($ln); }
        // Fit font size to box
        $boxH = $bottom - $textTop;
        $fontSize = ih_gd_fit_font($fontPath, $baseLines, $maxWidth, $boxH, $fontSize, $lineSpacing);
        // Now wrap (for any exceptionally long non label lines)
        foreach ($baseLines as $ln) {
            $wrapped = ih_wrap_text($fontPath, $fontSize, $ln, $maxWidth);
            foreach ($wrapped as $wln) $allLines[] = $wln;
        }
    } else {
        // No TTF font found: use built-in font (ASCII only) as last resort
        foreach ($lines as $ln) $allLines[] = ih_bidi_rtl_line($ln);
    }
    $lineHeight = (int)($fontSize * $lineSpacing);
    $totalHeight = count($allLines) * $lineHeight;
    $y = (int)(($textTop + $bottom) / 2 - $totalHeight / 2 + $fontSize);
    foreach ($allLines as $ln) {
        if ($fontPath && is_readable($fontPath)) {
            // center x
            $bbox = imagettfbbox($fontSize, 0, $fontPath, $ln);
            $tw = abs($bbox[4] - $bbox[0]);
            $x = (int)(($left + $right) / 2 - $tw / 2);
            imagettftext($src, $fontSize, 0, $x, (int)$y, $textColor, $fontPath, $ln);
        } else {
            // built-in GD font 4
            $fw = imagefontwidth(4) * strlen($ln);
            $fh = imagefontheight(4);
            $x = (int)(($left + $right) / 2 - $fw / 2);
            imagestring($src, 4, $x, (int)($y - $fh), $ln, $textColor);
        }
        $y += $lineHeight;
    }
    // Save
    imagejpeg($src, $output_path, 90);
    imagedestroy($src);
    return true;
}

function build_service_card_text($service, $server_display_name, $used_gb, $total_gb, $days_remaining) {
    $sanitize = function($s) {
        // Avoid bidi issues: replace parentheses and slash
        $s = str_replace(['(',')'], [' - ', ''], $s);
        $s = preg_replace('/\s+/', ' ', $s);
        return trim($s);
    };
    $lines = [];
    $service_name = $sanitize($service['display'] ?? ($service['name'] ?? ''));
    $loc = $sanitize($server_display_name);
    $lines[] = 'سرویس: ' . $service_name;
    $lines[] = 'لوکیشن: ' . $loc;
    if ($total_gb > 0) {
        $lines[] = 'مصرف: ' . number_format($used_gb, 2) . ' از ' . number_format($total_gb, 2) . ' GB';
    } else {
        $lines[] = 'مصرف: نامشخص';
    }
    $lines[] = 'باقیمانده: ' . max(0, (int)$days_remaining) . ' روز';
    return $lines;
}

// Normalize a server display name to an English-friendly form for the image
function ih_clean_location_en($display_name) {
    $s = (string)$display_name;
    // drop anything after a dash or parentheses (e.g., " - نیم بها")
    $s = preg_replace('/\s*[-–—]\s*.*$/u', '', $s);
    $s = preg_replace('/\s*\(.+\)$/u', '', $s);
    $s = trim($s);
    // common Persian -> English mappings (extend as needed)
    $map = [
        'ترکیه' => 'Turkey',
        'آلمان' => 'Germany',
        'امارات' => 'UAE',
        'ایران'  => 'Iran',
        'فرانسه'=> 'France',
        'هلند'  => 'Netherlands',
        'انگلستان' => 'UK',
        'بریتانیا' => 'UK',
        'آمریکا' => 'USA',
        'کانادا' => 'Canada',
        'روسیه' => 'Russia',
        'اروپا'  => 'Europe',
        'نیم بها' => '',
    ];
    foreach ($map as $fa => $en) {
        $s = preg_replace('/' . preg_quote($fa, '/') . '/u', $en, $s);
    }
    // final trim and collapse spaces
    $s = preg_replace('/\s+/', ' ', trim($s));
    return $s;
}

// English-only variant used by the bot when Latin output is desired
function build_service_card_text_en($service, $server_display_name, $used_gb, $total_gb, $days_remaining) {
    $sanitize = function($s) {
        $s = str_replace(['(',')','/','\\'], [' - ','','',''], (string)$s);
        $s = preg_replace('/\s+/', ' ', $s);
        return trim($s);
    };
    $lines = [];
    $service_name = $sanitize($service['display'] ?? ($service['name'] ?? ''));
    $loc = ih_clean_location_en($sanitize($server_display_name));
    $lines[] = 'Service: ' . $service_name;
    $lines[] = 'Location: ' . $loc;
    if ($total_gb > 0) {
        $lines[] = 'Usage: ' . number_format((float)$used_gb, 2) . ' / ' . number_format((float)$total_gb, 2) . ' GB';
    } else {
        $lines[] = 'Usage: Unlimited';
    }
    $lines[] = 'Days remaining: ' . max(0, (int)$days_remaining);
    return $lines;
}

?>
