<?php
require_once __DIR__ . '/../config.php';

if (!function_exists('ensure_referral_schema')) {
    function ensure_referral_schema(mysqli $connect): void {
        try { $connect->query("ALTER TABLE user ADD COLUMN referral_code VARCHAR(32) NULL"); } catch (Throwable $e) {}
        try { $connect->query("ALTER TABLE user ADD UNIQUE KEY uq_user_referral_code (referral_code)"); } catch (Throwable $e) {}
        try { $connect->query("ALTER TABLE user ADD COLUMN referred_by BIGINT NULL"); } catch (Throwable $e) {}
        try { $connect->query("ALTER TABLE user ADD INDEX idx_user_referred_by (referred_by)"); } catch (Throwable $e) {}
        try { $connect->query("ALTER TABLE user ADD COLUMN referral_opt_out TINYINT(1) NOT NULL DEFAULT 0"); } catch (Throwable $e) {}

        $connect->query("CREATE TABLE IF NOT EXISTS referral_rewards (
            id INT AUTO_INCREMENT PRIMARY KEY,
            referrer_id BIGINT NOT NULL,
            referred_user_id BIGINT NOT NULL,
            order_id INT NOT NULL,
            discount_code VARCHAR(64) NOT NULL,
            discount_id INT NULL,
            created_at TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
            UNIQUE KEY uq_referred_user (referred_user_id),
            KEY idx_referrer (referrer_id)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci");
        try { $connect->query("ALTER TABLE referral_rewards ADD INDEX idx_discount_id (discount_id)"); } catch (Throwable $e) {}
        try { $connect->query("ALTER TABLE referral_rewards ADD CONSTRAINT fk_referral_rewards_order FOREIGN KEY (order_id) REFERENCES orders(id) ON DELETE CASCADE"); } catch (Throwable $e) {}

        try {
            $connect->query("INSERT IGNORE INTO messages (message_key, message_value) VALUES
                ('referral_share_message', '🎁 <b>برنامه دعوت دوستان</b>\\n\\nلینک اختصاصی شما:\\n<code>{referral_link}</code>\\n\\nکد دعوت: <code>{referral_code}</code>\\n\\nبا دعوت هر دوست و انجام اولین خرید او، یک کد تخفیف {reward_value} ({reward_type_label}) دریافت می‌کنید.\\n\\n👥 دعوت‌شدگان: {referral_total_invited}\\n🏆 پاداش‌های دریافت‌شده: {referral_success_count}\\n⏳ دعوت‌های در انتظار: {referral_pending_count}'),
                ('referral_reward_notification', '🎉 تبریک! دوست شما {referral_user_id} اولین خرید خود را انجام داد.\\n\\n🎟️ کد تخفیف هدیه شما:\\n<code>{discount_code}</code>\\n\\n📌 مقدار تخفیف: {reward_value} ({reward_type_label})\\n📦 حداقل مبلغ خرید: {min_order_amount}\\n⏰ تاریخ انقضا: {expires_at}\\n🏆 مجموع پاداش‌های شما: {reward_count}\\n\\nبرای استفاده کافی است هنگام خرید کد را وارد کنید.'),
                ('referral_disabled_message', '⚠️ برنامه دعوت دوستان در حال حاضر فعال نیست.'),
                ('referral_welcome_referred', '🎉 با دعوت دوست‌تان به ربات پیوستید! پس از اولین خرید شما، {referrer_id} یک هدیه ویژه دریافت می‌کند.')
            ");
        } catch (Throwable $e) {}
    }
}

if (!function_exists('referral_get_setting')) {
    function referral_get_setting(mysqli $connect, string $key, $default = null) {
        $stmt = $connect->prepare("SELECT value FROM settings WHERE `key` = ? LIMIT 1");
        $stmt->bind_param('s', $key);
        $stmt->execute();
        $res = $stmt->get_result();
        $row = $res->fetch_assoc();
        $stmt->close();
        return $row['value'] ?? $default;
    }
}

if (!function_exists('referral_get_settings')) {
    function referral_get_settings(mysqli $connect): array {
        $keys = [
            'referral_status',
            'referral_reward_type',
            'referral_reward_value',
            'referral_reward_min_order',
            'referral_reward_expires_days',
            'referral_reward_prefix',
            'referral_reward_limit',
        ];
        $placeholders = implode(',', array_fill(0, count($keys), '?'));
        $stmt = $connect->prepare("SELECT `key`, `value` FROM settings WHERE `key` IN ($placeholders)");
        $stmt->bind_param(str_repeat('s', count($keys)), ...$keys);
        $stmt->execute();
        $res = $stmt->get_result();
        $settings = [];
        while ($row = $res->fetch_assoc()) {
            $settings[$row['key']] = $row['value'];
        }
        $stmt->close();

        $status = ($settings['referral_status'] ?? 'off') === 'on' ? 'on' : 'off';
        $reward_type = strtolower($settings['referral_reward_type'] ?? 'fixed');
        if (!in_array($reward_type, ['fixed', 'percent'], true)) {
            $reward_type = 'fixed';
        }
        $reward_value = (float)($settings['referral_reward_value'] ?? 0);
        if ($reward_type === 'percent') {
            $reward_value = max(1, min(100, $reward_value));
        } else {
            $reward_value = max(0, $reward_value);
        }
        $min_order = (int)($settings['referral_reward_min_order'] ?? 0);
        $expires_days = max(0, (int)($settings['referral_reward_expires_days'] ?? 0));
        $prefix = strtoupper(preg_replace('/[^A-Z0-9]/i', '', (string)($settings['referral_reward_prefix'] ?? 'REF')));
        $prefix = substr($prefix, 0, 12);
        if ($prefix === '') {
            $prefix = 'REF';
        }
        $limit = max(1, (int)($settings['referral_reward_limit'] ?? 1));

        return [
            'status' => $status,
            'reward_type' => $reward_type,
            'reward_value' => $reward_value,
            'reward_min_order' => $min_order,
            'reward_expires_days' => $expires_days,
            'reward_prefix' => $prefix,
            'reward_limit' => $limit,
        ];
    }
}

if (!function_exists('referral_is_enabled')) {
    function referral_is_enabled(mysqli $connect): bool {
        static $cache = null;
        if ($cache === null) {
            $cache = referral_get_settings($connect);
        }
        return ($cache['status'] ?? 'off') === 'on';
    }
}

if (!function_exists('referral_generate_random_code')) {
    function referral_generate_random_code(int $length = 8): string {
        $length = max(4, min(32, $length));
        try {
            $raw = bin2hex(random_bytes($length));
        } catch (Throwable $e) {
            $raw = md5(uniqid((string)mt_rand(), true));
        }
        $code = strtoupper(substr($raw, 0, $length));
        return $code;
    }
}

if (!function_exists('referral_generate_unique_user_code')) {
    function referral_generate_unique_user_code(mysqli $connect): string {
        do {
            $code = referral_generate_random_code(8);
            $stmt = $connect->prepare("SELECT id FROM user WHERE referral_code = ? LIMIT 1");
            $stmt->bind_param('s', $code);
            $stmt->execute();
            $exists = $stmt->get_result()->fetch_assoc();
            $stmt->close();
        } while ($exists);
        return $code;
    }
}

if (!function_exists('referral_ensure_user_code')) {
    function referral_ensure_user_code(mysqli $connect, string $user_id): string {
        $stmt = $connect->prepare("SELECT referral_code FROM user WHERE id = ? LIMIT 1");
        $stmt->bind_param('s', $user_id);
        $stmt->execute();
        $res = $stmt->get_result();
        $row = $res->fetch_assoc();
        $stmt->close();
        if (!empty($row['referral_code'])) {
            return $row['referral_code'];
        }
        $code = referral_generate_unique_user_code($connect);
        $stmt = $connect->prepare("UPDATE user SET referral_code = ? WHERE id = ? AND (referral_code IS NULL OR referral_code = '')");
        $stmt->bind_param('ss', $code, $user_id);
        $stmt->execute();
        $stmt->close();
        return $code;
    }
}

if (!function_exists('referral_resolve_referrer')) {
    function referral_resolve_referrer(mysqli $connect, ?string $payload, string $user_id): ?string {
        if (!$payload || !referral_is_enabled($connect)) {
            return null;
        }
        $code = strtoupper(trim($payload));
        $code = preg_replace('/[^A-Z0-9_-]/i', '', $code);
        if ($code === '') {
            return null;
        }
        $stmt = $connect->prepare("SELECT id FROM user WHERE referral_code = ? LIMIT 1");
        $stmt->bind_param('s', $code);
        $stmt->execute();
        $res = $stmt->get_result();
        $row = $res->fetch_assoc();
        $stmt->close();
        if (!$row) {
            return null;
        }
        $referrer_id = (string)$row['id'];
        if ($referrer_id === $user_id) {
            return null;
        }
        return $referrer_id;
    }
}

if (!function_exists('referral_assign_referred_by')) {
    function referral_assign_referred_by(mysqli $connect, string $user_id, string $referrer_id): bool {
        if ($user_id === $referrer_id) {
            return false;
        }
        $stmt = $connect->prepare("UPDATE user SET referred_by = ? WHERE id = ? AND (referred_by IS NULL OR referred_by = '' OR referred_by = '0')");
        $stmt->bind_param('ss', $referrer_id, $user_id);
        $stmt->execute();
        $affected = $stmt->affected_rows > 0;
        $stmt->close();
        return $affected;
    }
}

if (!function_exists('referral_generate_unique_discount_code')) {
    function referral_generate_unique_discount_code(mysqli $connect, string $prefix): string {
        $prefix = strtoupper(preg_replace('/[^A-Z0-9]/i', '', $prefix));
        $prefix = $prefix ? $prefix . '-' : '';
        do {
            $code = $prefix . referral_generate_random_code(6);
            $stmt = $connect->prepare("SELECT id FROM discount_codes WHERE code = ? LIMIT 1");
            $stmt->bind_param('s', $code);
            $stmt->execute();
            $exists = $stmt->get_result()->fetch_assoc();
            $stmt->close();
        } while ($exists);
        return $code;
    }
}

if (!function_exists('referral_record_reward')) {
    function referral_record_reward(mysqli $connect, string $referrer_id, string $referred_user_id, int $order_id, string $discount_code, ?int $discount_id = null): void {
        $stmt = $connect->prepare("INSERT INTO referral_rewards (referrer_id, referred_user_id, order_id, discount_code, discount_id) VALUES (?, ?, ?, ?, ?) ON DUPLICATE KEY UPDATE order_id = VALUES(order_id), discount_code = VALUES(discount_code), discount_id = VALUES(discount_id)");
        $stmt->bind_param('ssiis', $referrer_id, $referred_user_id, $order_id, $discount_code, $discount_id);
        $stmt->execute();
        $stmt->close();
    }
}

if (!function_exists('referral_format_currency')) {
    function referral_format_currency($amount): string {
        return number_format((int)$amount) . ' تومان';
    }
}

if (!function_exists('referral_format_datetime')) {
    function referral_format_datetime(?string $datetime): string {
        if (empty($datetime)) {
            return 'بدون انقضا';
        }
        try {
            $dt = new DateTime($datetime, new DateTimeZone('Asia/Tehran'));
        } catch (Throwable $e) {
            try { $dt = new DateTime($datetime); } catch (Throwable $ex) { return $datetime; }
        }
        if (function_exists('jdate')) {
            return jdate('Y/m/d H:i', $dt->getTimestamp());
        }
        return $dt->format('Y-m-d H:i');
    }
}

if (!function_exists('referral_get_user_stats')) {
    function referral_get_user_stats(mysqli $connect, string $user_id): array {
        $stats = [
            'invited' => 0,
            'successful' => 0,
            'pending' => 0,
        ];
        $stmt = $connect->prepare("SELECT COUNT(*) AS cnt FROM user WHERE referred_by = ?");
        $stmt->bind_param('s', $user_id);
        $stmt->execute();
        $res = $stmt->get_result();
        $stats['invited'] = (int)($res->fetch_assoc()['cnt'] ?? 0);
        $stmt->close();

        $stmt = $connect->prepare("SELECT COUNT(*) AS cnt FROM referral_rewards WHERE referrer_id = ?");
        $stmt->bind_param('s', $user_id);
        $stmt->execute();
        $res = $stmt->get_result();
        $stats['successful'] = (int)($res->fetch_assoc()['cnt'] ?? 0);
        $stmt->close();

        $stats['pending'] = max(0, $stats['invited'] - $stats['successful']);
        return $stats;
    }
}

if (!function_exists('referral_get_bot_username')) {
    function referral_get_bot_username(mysqli $connect): ?string {
        $username = referral_get_setting($connect, 'bot_username', '');
        if (!empty($username)) {
            return $username;
        }
        if (!function_exists('bot')) {
            return null;
        }
        $response = bot('getMe');
        $username = $response['result']['username'] ?? '';
        if (!empty($username)) {
            $username = trim($username);
            $stmt = $connect->prepare("INSERT INTO settings (`key`, `value`) VALUES ('bot_username', ?) ON DUPLICATE KEY UPDATE `value` = VALUES(`value`)");
            $stmt->bind_param('s', $username);
            $stmt->execute();
            $stmt->close();
            return $username;
        }
        return null;
    }
}

if (!function_exists('referral_build_link')) {
    function referral_build_link(mysqli $connect, string $code): ?string {
        $username = referral_get_bot_username($connect);
        if (!$username) {
            return null;
        }
        return 'https://t.me/' . $username . '?start=' . urlencode($code);
    }
}

if (!function_exists('referral_create_discount_reward')) {
    function referral_create_discount_reward(mysqli $connect, array $settings, array $order, string $referrer_id, string $referred_user_id): ?array {
        $type = $settings['reward_type'] ?? 'fixed';
        $value = (float)($settings['reward_value'] ?? 0);
        if ($value <= 0) {
            return null;
        }
        if ($type === 'percent') {
            $value = max(1, min(100, $value));
        } else {
            $type = 'fixed';
            $value = max(1, $value);
        }
        $prefix = $settings['reward_prefix'] ?? 'REF';
        $code = referral_generate_unique_discount_code($connect, $prefix);
        $limit = max(1, (int)($settings['reward_limit'] ?? 1));
        $per_user_limit = $limit;
        $min_order = max(0, (int)($settings['reward_min_order'] ?? 0));
        $min_order_param = $min_order > 0 ? $min_order : null;
        $expires_days = max(0, (int)($settings['reward_expires_days'] ?? 0));
        $expires_at = null;
        if ($expires_days > 0) {
            try {
                $dt = new DateTime('now', new DateTimeZone('Asia/Tehran'));
                $dt->modify('+' . $expires_days . ' days');
                $expires_at = $dt->format('Y-m-d H:i:s');
            } catch (Throwable $e) {
                $expires_at = null;
            }
        }
        $note = 'Referral reward for user ' . $referred_user_id;
        $stmt = $connect->prepare("INSERT INTO discount_codes (code, type, value, max_uses, per_user_limit, min_order_amount, expires_at, is_active, note) VALUES (?, ?, ?, ?, ?, ?, ?, 1, ?)");
        $value_param = ($type === 'percent') ? $value : (float)$value;
        $stmt->bind_param('ssdiiiss', $code, $type, $value_param, $limit, $per_user_limit, $min_order_param, $expires_at, $note);
        $stmt->execute();
        $discount_id = $stmt->insert_id;
        $stmt->close();

        return [
            'code' => $code,
            'discount_id' => $discount_id ?: null,
            'type' => $type,
            'value' => $value,
            'value_text' => $type === 'percent' ? number_format($value) . '%' : referral_format_currency($value),
            'type_label' => $type === 'percent' ? 'درصد' : 'تومان',
            'min_order' => $min_order,
            'min_order_text' => $min_order > 0 ? referral_format_currency($min_order) : 'بدون محدودیت',
            'expires_at' => $expires_at,
            'expires_at_text' => referral_format_datetime($expires_at),
        ];
    }
}

if (!function_exists('referral_process_successful_order')) {
    function referral_process_successful_order(mysqli $connect, array $order): void {
        if (!referral_is_enabled($connect)) {
            return;
        }
        $order_id = (int)($order['id'] ?? 0);
        $user_id = (string)($order['user_id'] ?? '');
        if ($order_id <= 0 || $user_id === '') {
            return;
        }
        if (($order['type'] ?? 'new') === 'exten') {
            return;
        }
        $stmt = $connect->prepare("SELECT referred_by FROM user WHERE id = ? LIMIT 1");
        $stmt->bind_param('s', $user_id);
        $stmt->execute();
        $res = $stmt->get_result();
        $user_row = $res->fetch_assoc();
        $stmt->close();
        $referrer_id = $user_row['referred_by'] ?? null;
        if (empty($referrer_id)) {
            return;
        }
        $stmt = $connect->prepare("SELECT id FROM referral_rewards WHERE referred_user_id = ? LIMIT 1");
        $stmt->bind_param('s', $user_id);
        $stmt->execute();
        $already = $stmt->get_result()->fetch_assoc();
        $stmt->close();
        if ($already) {
            return;
        }
        $stmt = $connect->prepare("SELECT COUNT(*) AS cnt FROM orders WHERE user_id = ? AND status='approved'");
        $stmt->bind_param('s', $user_id);
        $stmt->execute();
        $res = $stmt->get_result();
        $count = (int)($res->fetch_assoc()['cnt'] ?? 0);
        $stmt->close();
        if ($count > 1) {
            return;
        }

        $settings = referral_get_settings($connect);
        if (($settings['status'] ?? 'off') !== 'on') {
            return;
        }
        $reward = referral_create_discount_reward($connect, $settings, $order, (string)$referrer_id, $user_id);
        if (!$reward) {
            return;
        }
        referral_record_reward($connect, (string)$referrer_id, $user_id, $order_id, $reward['code'], $reward['discount_id']);

        $stmt = $connect->prepare("SELECT COUNT(*) AS cnt FROM referral_rewards WHERE referrer_id = ?");
        $stmt->bind_param('s', $referrer_id);
        $stmt->execute();
        $res = $stmt->get_result();
        $total_rewards = (int)($res->fetch_assoc()['cnt'] ?? 0);
        $stmt->close();

        if (function_exists('getMessage')) {
            $placeholders = [
                'discount_code' => $reward['code'],
                'reward_value' => $reward['value_text'],
                'reward_type_label' => $reward['type_label'],
                'min_order_amount' => $reward['min_order_text'],
                'expires_at' => $reward['expires_at_text'],
                'referral_user_id' => $user_id,
                'reward_count' => number_format($total_rewards),
            ];
            $message = getMessage($connect, 'referral_reward_notification', $placeholders);
            if (!empty($message) && function_exists('sendMessage')) {
                global $start;
                sendMessage($referrer_id, $message, $start ?? null, 'html');
            }
        }
    }
}
